/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */

package org.jvnet.glassfish.comms.sipagent.ui;

import org.jvnet.glassfish.comms.sipagent.backend.BackendException;
import org.jvnet.glassfish.comms.sipagent.backend.Controller;
import org.jvnet.glassfish.comms.sipagent.backend.ControllerFactory;
import org.jvnet.glassfish.comms.sipagent.model.ModelListener;
import org.jvnet.glassfish.comms.sipagent.model.SipMessage;
import org.jvnet.glassfish.comms.sipagent.model.SipRequest;
import org.jvnet.glassfish.comms.sipagent.model.SipResponse;
import org.jvnet.glassfish.comms.sipagent.support.Constants;
import org.jvnet.glassfish.comms.sipagent.support.LogSupport;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *  ControllerAgent handles the GUI communication with the data model. 
 *
 * The ControllerAgent offers a prepareRequest(String aText) method to the
 *  GUI, for creating a SipRequest based on input text. this will always
 * result in a valid SipMessage, be it the actual message or a default.
 *
 * @author elnyvbo
 */
public class ControllerAgent implements ModelListener {
    
    private static Logger theirLogger =
        Logger.getLogger("org.jvnet.glassfish.comms.sipagent.ui.ControllerAgent");
    
    private Controller itsController = null;
    private AgentMainJPanel itsPanel = null;
    private File itsTrafficLog = null;
    
    /**
     * Creates a new instance of ControllerAgent
     */
    public ControllerAgent(AgentMainJPanel aPanel, Constants aConstants) {
        theirLogger.finest("-----> ControllerAgent ----->");
        itsController = ControllerFactory.instance().createController(aConstants);   
        itsController.registerListener(this);
        itsPanel = aPanel;
        
        theirLogger.finer("Created controller " + itsController);
        
        // open a log file for sip traffic, if one has been specified
        if ("true".equals(
            aConstants.getProperty(Constants.TRAFFIC_LOGGING_ENABLED))){
            String trafficlog = 
                aConstants.getProperty(Constants.LOG_FILE);
            initLogFile(trafficlog);
        }
        else {
            theirLogger.finer("SIP traffic logging disabled. Set the " 
                + "system property " + Constants.TRAFFIC_LOGGING_ENABLED 
                + " to \"true\" to enable SIP traffic logging, or select a "
                + "logfile in the GUI");
        }
        
        theirLogger.finest("<----- ControllerAgent -----");
    }
    
    /**
     *  Deletes the file with given name and opens it as new logfile. Any
     *  exceptions are caught and reported to java.util.logging.
     */
    public void initLogFile(String aFileName){
         try {
            itsTrafficLog = new File(aFileName);
            itsTrafficLog.delete();
            itsTrafficLog.createNewFile();
            theirLogger.finer("Created new trafficlog: " + aFileName);
        } catch (Throwable e) {
            itsTrafficLog = null;
            theirLogger.severe("ERROR: Failed opening traffic log: " +
                LogSupport.stringify(e));
        }
    }
    
    /**
     *  Close connection with the backend.
     */
    public void close() {
        theirLogger.finest("----- close ----->");
        if (itsController != null) {
            theirLogger.finest("Closing controller...");
            itsController.close();
        }
        theirLogger.finest("<----- close -----");
    }
    
    /**
     *  Used to prepare a request in the data model based on a text blob.
     *  If parsing of the blob fails, the agent will construct a default 
     *  message.
     */
    public RequestBean prepareRequest(String aRequestText){
        boolean initialize = false;
        theirLogger.finest("----- prepareRequest ----->");
        // First try to create a request based on the text the GUI passed in
        SipRequest msg = null;
        try {
            msg = itsController.createRequest(aRequestText);
        } catch (BackendException e) {
            theirLogger.log(Level.FINER,null,e);
            theirLogger.warning(
                "Caught exception while trying to create a request based on the" 
                + " user input. Created a new default SIP request.");
            // no, so just create a new default request.
            msg = itsController.createRequest();
            initialize = true;
        }
        

        RequestBean result = new RequestBean(msg);
        if (initialize) {
            result.initialize();
        }
        theirLogger.finest("<----- prepareRequest -----");
        return result;
    }
    
    /**
     *  Create a response bean based on the input text. The agent will try to
     *  parse the text as a request, and if that succeeds, create a resopnse 
     *  based on it.
     */
    public ResponseBean prepareResponseFromRequest(String aRequestText) {
        theirLogger.finest("----- prepareResponseFromRequest() ----->");
        
        // TODO if request preparation fails this may result in a illogical 
        // response. No big deal though, user can still change it as he pleases.
        RequestBean req = this.prepareRequest(aRequestText);
        ResponseBean resp = new ResponseBean(req);
        
        theirLogger.finest("<----- prepareResponseFromRequest() -----");
        return resp;
    }
    
    public ResponseBean prepareResponse(String aResponseText) {
        theirLogger.finest("----- prepareResponse() ----->");
        SipResponse resp = null;
        try {
            resp = itsController.createResponse(aResponseText);    
        }
        catch (BackendException e){
            theirLogger.log(Level.FINER,
                "Unable to prepare a response based on the text:\n"
                + aResponseText, e);
        }
        ResponseBean bean = new ResponseBean(resp);
        theirLogger.finest("<----- prepareResponse() -----");
        return bean;
    }
    
    /**
     *  Try to create a request based on the input text. If that fails, try to
     *  create a response. 
     */
    public void sendMessage(String aText) {
        theirLogger.finest("----- sendMessage ----->");
        // Try to create a request based on the text the GUI passed in.
        SipMessage msg = null;
        try {
            msg = itsController.createRequest(aText);
        }
        catch (BackendException e1){
            // TODO: clean up yucky use of exceptions for flow-of-control
            // not a request, was it a response?
            try {
                msg = itsController.createResponse(aText);
            }
            catch (BackendException e2){    
                // nope, then we give up.
                theirLogger.log(Level.FINER,aText,e2);
                throw new RuntimeException("Invalid text format!", e2);    
            }
        }
        // send resulting SipMessage to backend.
        itsController.sendMessage(msg);
        
        theirLogger.finest("<----- sendMessage -----");
    }

    /**
     *  Method dictated by the ModelListener interface.
     *
     *  Called by the backend when a new message arrives in the data model.
     *  ControllerAgent will briefly analyse the message and update the GUI.
     */
    public void incomingMessage(SipMessage aMessage) {
        
        // write message in traffic log TODO move to separate modellistener
        if (itsTrafficLog != null){
            try {
                // append to the traffic log
                BufferedWriter out = 
                    new BufferedWriter(new FileWriter(itsTrafficLog, true));
                out.write("\n----- MESSAGE RECEIVED -----\n" + aMessage );
                out.close();
            } catch (IOException e) {
              theirLogger.warning("WARNING: Could not write to trafficlog!")  ;
            }
        }
        
        // Create Bean and add that to the history.
        if (aMessage instanceof SipRequest){
            RequestBean bean = new RequestBean((SipRequest)aMessage);
            itsPanel.addToMessageHistory(bean);
        }
        else if (aMessage instanceof SipResponse){
            ResponseBean bean = new ResponseBean((SipResponse)aMessage);
            itsPanel.addToMessageHistory(bean);
        }
    }

    /**
     *  Method dictated by the ModelListener interface.
     *
     *  Called by the backend when a message, which is a retransmission of an 
     *  earlier message, arrives in the data model.
     */
    public void reTransmission(SipMessage aMessage) {
        // TODO: update the GUI with a number behind the original incoming message
    }

    /**
     *  Method dictated by the ModelListener interface.
     *  Called by the backend when a message from the data model has been 
     *  sent out.
     */
    public void outgoingMessage(SipMessage aMessage) {
        // write message in traffic log TODO move to separate modellistener
        if (itsTrafficLog != null){
            try {
                // append to the traffic log
                BufferedWriter out = 
                    new BufferedWriter(new FileWriter(itsTrafficLog, true));
                out.write("\n----- MESSAGE SENT -----\n" + aMessage );
                out.close();
            } catch (IOException e) {
              theirLogger.warning("WARNING: Could not write to trafficlog!")  ;
            }
        }
        
        // Create Bean and add that to the history.
        if (aMessage instanceof SipRequest){
            RequestBean bean = new RequestBean((SipRequest)aMessage);
            itsPanel.addToMessageHistory(bean);
        }
        else if (aMessage instanceof SipResponse){
            ResponseBean bean = new ResponseBean((SipResponse)aMessage);
            itsPanel.addToMessageHistory(bean);
        }
    }
    

}
