/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.sipagent.ui;

import java.util.HashMap;
import org.jvnet.glassfish.comms.sipagent.model.SipRequest;
import org.jvnet.glassfish.comms.sipagent.support.Constants;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *  Bean class for offering a simple interface to the GUI for constructing a SIP
 *  request. The model classes cannot be used for this purpose; those were 
 *  originally designed to be a part of the sailfin SIP Container, which makes
 *  it impossible to manipulate system headers such as Contact, From and To.
 *
 *  This bean class can be manipulated freely, converted to String when done
 *  and then fed into the SipParser for constructing a 'neat' SIP message.
 *
 * @author elnyvbo
 */
public class RequestBean extends MessageBean {
    
    private static Logger theirLogger =
        Logger.getLogger("org.jvnet.glassfish.comms.sipagent.ui.RequestBean");

    private String itsURI = "";

    
    /** Creates a new instance of RequestMap */
    public RequestBean(SipRequest aMsg) {
        theirLogger.finest("----- RequestBean() ----->");
        
        // retrieve stuff from aMsg and cache locally
        setURI(aMsg.getURI());
        setFromName(aMsg.getFromName()); 
        setFromURI(aMsg.getFromURI());
        setToName(aMsg.getToName()); 
        setToURI(aMsg.getToURI());
        setToTag(aMsg.getToTag());
        setFromTag(aMsg.getFromTag());
        
        // copy headers
        Iterator headers = aMsg.getImpl().getHeaderNames();
        while (headers.hasNext()){
            String header = (String)headers.next();
            // get values
            ListIterator values = aMsg.getImpl().getHeaders(header);
            while (values.hasNext()){
                this.addHeaderValue(header,(String)values.next());
            }
        }
        
        setMethod(aMsg.getMethod());
        setContent(aMsg.getContent());
        theirLogger.finest("<----- RequestBean() -----");
    }
    
    /**
     *  Creates a RequestBean that represents a subsequent request,
     *  based on a ResponseBean. CallID, CSeq headers are copied. 
     * 
     *  Contact header is used as new request URI.
     */
    public RequestBean(ResponseBean aResponse){
         // copy method, from and to
        setMethod(aResponse.getMethod());
        setFromName(aResponse.getFromName());
        setFromURI(aResponse.getFromURI());
        setToName(aResponse.getToName());
        setToURI(aResponse.getToURI());
        setURI(aResponse.getToURI());
        setToTag(aResponse.getToTag());
        setFromTag(aResponse.getFromTag());
        
        
        // copy contact header from response and use as request URI
        // (typically when sending ACK or BYE to an INVITE)
        String[] contacts = aResponse.getHeaderValues(Constants.CONTACT);
        if ((contacts != null) && (contacts.length > 0)){
            // just use the first one
            String contact = contacts[0];
            // strip "<" and ">"
            String strippedContact = 
                contact.replace(">", "").replace("<","");
            setURI(strippedContact);
        }
        
        // copy other relevant headers
        setHeaderValues(
            Constants.CALL_ID, aResponse.getHeaderValues(Constants.CALL_ID));
        setHeaderValues(
            Constants.CSEQ, aResponse.getHeaderValues(Constants.CSEQ));
        setHeaderValues(Constants.CONTENT_LENGTH, new String[]{""+0});
        setHeaderValues(Constants.VIA, aResponse.getHeaderValues(Constants.VIA));
    }
    
    public String getURI(){
        return itsURI;
    }
    public void setURI(String aURI){
        itsURI = aURI;
    }
    
    /**
     * Use this method to create new CallId, from-Tag, to-Tag, and CSEQ numbers
     */
    void initialize(){
        
        setHeaderValues(
            Constants.CALL_ID, new String[]{"" + System.currentTimeMillis()});
        setFromTag("" + System.currentTimeMillis());
        setToTag("");
        
    }
     
    static Map<String,Integer> cSeqMap = new HashMap<String,Integer>(7);
    
    void updateCSeq(){
        // retrieve the method part of the CSEQ header as the index to the counter
        String cSeqOld = getHeaderValues(Constants.CSEQ)[0];
        
        String cSeqMethodStr = cSeqOld.substring(cSeqOld.indexOf(" "));
        Integer cSeq = cSeqMap.get(cSeqMethodStr);
        if (null == cSeq) {
            cSeq = Integer.valueOf(0);
        }
        cSeq = Integer.valueOf(cSeq.intValue() + 1);
        cSeqMap.put(cSeqMethodStr,cSeq);
        
        setHeaderValues(Constants.CSEQ, new String[]{cSeq.toString() + " " + cSeqMethodStr});
    }
    
    public String toString(){
        final String SP = " ";
        final String CRLF = Constants.CRLF;
        final String FROM = Constants.FROM;
        final String TO = Constants.TO;
        
        final String FROMTAG = this.getFromTag();
        final String TOTAG = this.getToTag();
        
        StringBuilder result = new StringBuilder(
            getMethod() + SP + getURI() + SP + "SIP/2.0" + CRLF 
            + FROM + ": \"" + getFromName() + "\" <" + getFromURI() +  ">"); 
            
             
        // add tag if present
        if ((!"".equals(FROMTAG)) && (FROMTAG != null)){
            result.append(";tag=").append(FROMTAG);
        }
        
        result.append(    
            CRLF + TO + ": \"" + getToName() + "\" <" + getToURI() + ">");
                
        // add tag if present
        if ((!"".equals(TOTAG)) && (TOTAG != null)){
            result.append(";tag=").append(TOTAG);
        }
        
        result.append(CRLF);
        
        Iterator<String> headers = getHeaders();
        while (headers.hasNext()){
            String header = headers.next();
            // skip from and to as we've just added those
            if (!(FROM.equals(header))&&!(TO.equals(header))){
                String[] values = getHeaderValues(header);
                for (String s : values){
                    result.append(header).append(": ").append(s).append(CRLF);                    
                }
            }
        }
        result.append(CRLF).append(getContent());
            
        return result.toString();
    }
}
