/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.enterprise.ee.web.sessmgmt;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;

import java.util.concurrent.*;
import java.util.concurrent.atomic.*;
import java.util.*;

import java.lang.reflect.Method;

import java.util.logging.Level;
import java.util.logging.Logger;
import com.sun.logging.LogDomains;

import net.jxta.document.*;
import net.jxta.endpoint.Message;
import net.jxta.endpoint.MessageElement;
import net.jxta.endpoint.MessageTransport;
import net.jxta.endpoint.TextDocumentMessageElement;
import net.jxta.impl.endpoint.router.EndpointRouter;
import net.jxta.impl.endpoint.router.RouteControl;
import net.jxta.impl.pipe.BlockingWireOutputPipe;
import net.jxta.peer.PeerID;
import net.jxta.peergroup.PeerGroup;
import net.jxta.pipe.OutputPipe;
import net.jxta.pipe.PipeService;
import net.jxta.protocol.PipeAdvertisement;
import net.jxta.protocol.RouteAdvertisement;

import com.sun.enterprise.web.ServerConfigLookup;

/**
 *
 * @author Larry White
 */
public class JxtaReplicationUnicastSender {

    private final static Level TRACE_LEVEL = Level.FINE;    
    public final static String LOGGER_MEM_REP 
        = ReplicationState.LOGGER_MEM_REP;
    
    private static final Logger _logger 
        = Logger.getLogger(LOGGER_MEM_REP); 
    
    private static final Logger _salogger = ReplicationUtil.getSALogger();
    private static final Logger _pipelogger = ReplicationUtil.getPipeLogger();    
    
    private static final int INITIAL_CAPACITY = 2048;
    
    private static final float LOAD_FACTOR = 0.75F;
    
    private static final int CONCURRENCY_LEVEL = 100;
    
    private DispatchThreadNoDupsAllowed dispatchThreadNoDupsAllowed 
        = new DispatchThreadNoDupsAllowed();
    
    private static final int NUMBER_OF_SESSIONS_PER_MESSAGE = 20;
    
    private static final int BULK_MESSAGE_LIMIT = 32*1024;
    
    private static AtomicInteger _messageIDCounter = new AtomicInteger(0);
    
    private static long DEFAULT_TIME_THRESHOLD = 100L;
    
    /**
     * number of failed attempts to send a unicast response
     */
    protected static AtomicLong _unicastMessageFailures = new AtomicLong(0);
    
    // private final int NTHREADS = (getNumberOfSenderThreads() * 2);
    private static final int NTHREADS = getNumberOfSenderThreads() / 2;

    private static final ThreadPoolExecutor exec;    
    
    private AtomicBoolean timeToChangeNoDupsAllowed = new AtomicBoolean(false); 
    
    private AtomicLong lastChangeTimeNoDupsAllowed = new AtomicLong(System.currentTimeMillis() - 1000L);
    
    AtomicReference currentMapNoDupsAllowed = new AtomicReference(new ConcurrentHashMap<ReplicationState, ReplicationState>(INITIAL_CAPACITY, LOAD_FACTOR, CONCURRENCY_LEVEL));
    
    private static LinkedBlockingQueue senderTaskQueue = null;
    
    String _instanceName = null;
    
    private Object dispatchThreadNoDupsAllowedMonitorObject = new Object();
    
    static final int COREPOOLSIZE = NTHREADS;
    static final int MAXPOOLSIZE = NTHREADS;
    static final int KEEPALIVETIME = 15;  //seconds that a thread remains idle before terminating.
    
    private static int getNumberOfSenderThreads() {
        //make equal to number of pipes - no less than 5
        ServerConfigLookup lookup = new ServerConfigLookup();
        int result = lookup.getNumberOfReplicationPipesFromConfig();
        if(result < 5) {
            result = 5;
        }
        return result;
    }     
    
    static {

        senderTaskQueue = new LinkedBlockingQueue<Runnable>();
        ThreadGroup threadGroup = new ThreadGroup(Thread.currentThread().getThreadGroup(), "Group ReplicationUnicastSender");
        exec = new ThreadPoolExecutor(COREPOOLSIZE, MAXPOOLSIZE,
                                      KEEPALIVETIME, TimeUnit.SECONDS,
                                      senderTaskQueue,
                                      new ReplicationSenderThreadFactory("ReplicationUnicastSenderExecutor", threadGroup));

        // Try to allow core threads to idle out. (Requires a 1.6 method)
        try {
            Method allowCoreThreadTimeOut = exec.getClass().getMethod("allowCoreThreadTimeOut", boolean.class);

            allowCoreThreadTimeOut.invoke(exec, Boolean.TRUE);
        } catch (Throwable ohWell) {
            // Our attempt failed.
            if (_logger.isLoggable(Level.FINEST)) {
                _logger.log(Level.FINEST, "Failed to enable 'allowCoreThreadTimeOut'", ohWell);
            }
        }        
    }
    
    /** Creates a new instance of JxtaReplicationUnicastSender */
    public JxtaReplicationUnicastSender(String instanceName) {
        this._instanceName = instanceName;      
    }
    
    /**
     * latency check counter
     */
    protected AtomicInteger _latencyCheckCounter = new AtomicInteger(-1);    
    
    private volatile int _latencyCount = -1;    
    
    /** gets the latency count limit */
    private int getLatencyCountLimit() {
        if(_latencyCount == -1) {
            ServerConfigLookup lookup = new ServerConfigLookup();
            _latencyCount = lookup.getLatencyCountPropertyFromConfig();
        }
        return _latencyCount;
    } 
    
    private int incrementLatencyCheckCount() {
        return (Math.abs(_latencyCheckCounter.incrementAndGet()));
    } 
    
    private boolean shouldWait(boolean wait, int latencyCount, int latencyCountLimit) {
        //System.out.println("shouldWait>>wait = " + wait + " latencyCount = " + latencyCount + " latencyCountLimit = " + latencyCountLimit);
        if(!wait && latencyCountLimit == 0) {
            return false;
        } else {
            return(wait || (latencyCount % latencyCountLimit) == 0);
        }
    }     
    
    public ReplicationState sendReplicationState(ReplicationState state, boolean wait) {
        replicateStateNoDupsAllowed(state, wait);
        
        return null;  //FIXME
    }      
    
    private void replicateStateNoDupsAllowed(ReplicationState state, boolean wait) {
        //System.out.println("JxtaReplicationUnicastSender>>replicateStateNoDupsAllowed:wait = " + wait);
        boolean needsToWait = false;
        int latencyCountLimit = this.getLatencyCountLimit();
        int latencyCount = incrementLatencyCheckCount();
        needsToWait = shouldWait(wait, latencyCount, latencyCountLimit);

        if(needsToWait) {
            state.setAckRequired(needsToWait);            
        }
        try {
            //this method also controls whether this thread waits or not
            boolean thisThreadShouldWait 
                = addToMapNoDups(currentMapNoDupsAllowed, state, needsToWait);
            //int requestCount = requestCounterNoDupsAllowed.incrementAndGet();
            //long lastChangeTime = lastChangeTimeNoDupsAllowed.get();
            if ( ((Map)currentMapNoDupsAllowed.get()).size() >= NUMBER_OF_SESSIONS_PER_MESSAGE) {
                boolean wakeupDispatcher = timeToChangeNoDupsAllowed.compareAndSet(false, true); //expect false  set  to true            
                if (wakeupDispatcher) {
                    dispatchThreadNoDupsAllowed.wakeup();
                }
            }
            
            if(thisThreadShouldWait) {
                //wait or latencyCountLimit hit case
                //block and wait for return message
                ReplicationState returnState = 
                    ReplicationResponseRepository.getEntry((String)state.getId());
                if (returnState == null) {
                    //System.out.println("JxtaReplicationUnicastSender>>replicateStateNoDupsAllowed timed out returning null");
                } else {
                    //System.out.println("ack received: id = " + state.getId());
                    //System.out.println("JxtaReplicationUnicastSender>>replicateStateNoDupsAllowed succeeded");
                }               
            } 
           
        } finally {
            //rLock.unlock();
        }        
        
    }    
    
    private boolean addToMapNoDups(AtomicReference map, ReplicationState state, boolean wait) {
        Map tempMap = (Map)map.get();
        boolean currentThreadShouldWait = wait;
        boolean shouldContinue = true;
        do{
            //start by assuming wait parameter will guide this
            currentThreadShouldWait = wait;
            //start by assuming there is no other thread waiting
            boolean threadAlreadyWaiting = false;
            //check if existing state has thread waiting
            ReplicationState currentState = (ReplicationState)tempMap.get(state);                
            if(currentState != null && currentState.isAckRequired()) {
                threadAlreadyWaiting = true;
                state.setAckRequired(true);
            } else { //no thread waiting yet 
                state.setAckRequired(wait);
                //only in this case is waiting room set up
                if(wait) {
                    LinkedBlockingQueue aQueue =
                        ReplicationResponseRepository.putEmptyQueueEntry(state);
                    currentThreadShouldWait = true;
                }
            }         
            
            //now unconditionally add the state
            //flush method will now check if sent
            tempMap.put(state, state);
            if (_logger.isLoggable(TRACE_LEVEL)) {
                _logger.log(TRACE_LEVEL, "addToMapNoDups added state id: "
                    + state.getId() + "[ver: " + state.getVersion() + "] to mapHashCode: " + 
                    System.identityHashCode(tempMap));
            }                
            if(tempMap == currentMapNoDupsAllowed.get()) {
                shouldContinue = false;
                if (_logger.isLoggable(TRACE_LEVEL)) {
                    _logger.log(TRACE_LEVEL, "addToMapNoDups exiting... ");
                }                 
            } else {
                tempMap = (Map)currentMapNoDupsAllowed.get();
            }
        } while (shouldContinue);
        return currentThreadShouldWait;
    }    
    
    //Called by the dispatcher
    void flushAllMessagesFromCurrentMapNoDupsAllowed(boolean waitForAck) {
        //System.out.println("flushAllMessagesFromCurrentMapNoDupsAllowed:begin");        
        Map oldMap = null;
        try {
            //wLock.lock();
            oldMap = (Map)currentMapNoDupsAllowed.get();
            currentMapNoDupsAllowed.set(new ConcurrentHashMap<ReplicationState, ReplicationState>(INITIAL_CAPACITY, LOAD_FACTOR, CONCURRENCY_LEVEL));
            //wLock.unlock();
            lastChangeTimeNoDupsAllowed.set(System.currentTimeMillis());
            if (_logger.isLoggable(TRACE_LEVEL)) {
                _logger.log(TRACE_LEVEL, "flushAllMessagesFromCurrentMapNoDupsAllowed flipped maps oldMapKey: "
                     + System.identityHashCode(oldMap) + "currentMapNoDupsAllowed.get()Key):"
                     + System.identityHashCode(currentMapNoDupsAllowed.get()));
            }             
            timeToChangeNoDupsAllowed.set(false);
        } finally {
        }
        
        //_logger.log(Level.INFO, ">>JxtaReplicationUnicastSender::flushAllMessagesFromCurrentMapNoDupsAllowed: " + oldMap.size());
        
        //Send sessions in currentMap into a message
        //System.out.println("flush full list size = " + oldMap.keySet().size());
        List<ReplicationState> list = new ArrayList<ReplicationState>(oldMap.size()+1);
        Iterator<ReplicationState> iter = oldMap.values().iterator();
        int totalMessageSize = 0;
        RouteAdvertisement routeAdv = null;
        boolean firstTime = true;
        //non order preserving
        while (iter.hasNext()) {            
            boolean needBulkAck = false;
            ReplicationState state = iter.next();
            if(firstTime) {
                firstTime = false;
                routeAdv = state.getRouteAdvertisement();
            }
            int stateSize = 0;
            if(state != null && !state.isSent()) {
                //if any state requires ack then this bulk msg requires ack
                if(state.isAckRequired()) {
                    needBulkAck = true;
                }
                if(state.getState() != null) {
                    stateSize = state.getState().length;
                }                 
            }                       
            if (totalMessageSize + stateSize > BULK_MESSAGE_LIMIT) {
                doThreadedCreateMessageAndSend(list, needBulkAck, routeAdv); 
                list = new ArrayList<ReplicationState>(oldMap.size()+1); 
                //createMessageAndSend(list, needBulkAck, null);
                //list.clear();
                totalMessageSize = 0;
            }
            if(state != null && !state.isSent()) {
                //this is because route advertisement is not serializable
                state.setRouteAdvertisement(null);
                state._instanceName = ReplicationUtil.getInstanceName();
                //mark state as being sent
                state.setSent(true);
                list.add(state);
            }
            totalMessageSize += stateSize;
        }       
        
        if (list.size() > 0) {
            boolean needBulkAck = doesListContainAckRequiredState(list);
            doThreadedCreateMessageAndSend(list, needBulkAck, routeAdv);
            //createMessageAndSend(list, needBulkAck, null);
            //list.clear();
        }
        oldMap.clear();
    }    
    
    private void doThreadedCreateMessageAndSend(List list, boolean needBulkAck, RouteAdvertisement routeAdv) {
        //System.out.println("doThreadedCreateMessageAndSend");
        BulkMessageSender bulkMessageSender 
            = new BulkMessageSender(list, needBulkAck, routeAdv);
        exec.execute(bulkMessageSender);
    }
    
    private void createMessageAndSend(List<ReplicationState> list, boolean waitForAck, RouteAdvertisement routeAdv) {
        List<String> ackIdsList = ReplicationState.extractAckIdsList(list);
        //displayList(ackIdsList, "ackIdsList");
        byte[] data = null;
        ByteArrayOutputStream bos = null;
        ObjectOutputStream oos = null;
        try {
            bos = new ByteArrayOutputStream();
            oos = new ObjectOutputStream(bos);
            oos.writeObject(list);
            oos.flush();
        } catch (IOException ioEx) {
            //FIXME
        } finally {
            if (oos != null) {
                try {
                    oos.flush(); oos.close();
                } catch (Exception ex) {
                    //Ignore
                }
            }
            
            if (bos != null) {
                try {
                    bos.flush();
                    data = bos.toByteArray();
                    bos.close();
                } catch (Exception ex) {
                    //FIXME
                }
            }
        }
        //_logger.log(Level.INFO, "<<JxtaReplicationUnicastSender::flushAllMessages: About to send " + list.size() + " messages: " + data.length + " bytes...");
        List<String> allIdsList = null;
        if (_logger.isLoggable(TRACE_LEVEL)) {            
            allIdsList = ReplicationState.extractAllIdsList(list);
            displayList(allIdsList, "ids about to be sent");            
        }        
        
        int bulkMsgId = Math.abs(_messageIDCounter.incrementAndGet());
        if (_salogger.isLoggable(Level.FINE)) {
            if (allIdsList == null) {
                allIdsList = ReplicationState.extractAllIdsList(list);
            }
            displayList(_salogger, allIdsList, "sending replication state in bulkMsgId:" + bulkMsgId + " ");
        }
        sendBulkMessage(bulkMsgId, ackIdsList, data, waitForAck, routeAdv);        
        //_logger.log(Level.INFO, "<<JxtaReplicationUnicastSender::flushAllMessages: DONE!!");
    }
    
    private ReplicationState sendBulkMessage(long msgID, List<String> ackIdsList, byte[] data, boolean wait, RouteAdvertisement routeAdv) {
        //System.out.println("sending bulk message: bulkMsgId = " + msgID);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("JxtaReplicationUnicastSender>>sendBulkMessage:wait:" + wait + " bulkMggId:" + msgID);
        }         
        ReplicationState returnState = null;
        if(!ReplicationHealthChecker.isOkToProceed()) {
            if (_salogger.isLoggable(Level.FINE)) {
                _salogger.finer("health check bypassing replication. bulkMsgId " + msgID + " not sent");
            }
            return returnState;
        }
        
        boolean ackRequired = wait;      
        //long startTime = System.currentTimeMillis();
        ReplicationState state = ReplicationState.createBulkReplicationState(msgID, ackIdsList, data, ackRequired);
        state.setRouteAdvertisement(routeAdv);
        if( (wait) 
            || ReplicationHealthChecker.isFlushThreadWaiting()) {
            LinkedBlockingQueue aQueue =
                    ReplicationResponseRepository.putEmptyQueueEntry(state);
        }
        //this will try to send on different pipes until timeout
        this.sendReplicationStateQueryResponse(state, _instanceName);          
        
        if(wait) {
            //block and wait for return message
            returnState = 
                ReplicationResponseRepository.getEntry((String)state.getId(), 5000L);
            /*
            long waitTime = System.currentTimeMillis() - startTime;
            if(waitTime > 15000L) {
                //System.out.println("wait time = " + waitTime);
            }
             */
        }
        return returnState;
    }
    
    private Message createMessage(ReplicationState state, boolean isResponse) {
        return ReplicationState.createMessage(state, isResponse);
    } 
    
    public ReplicationState sendReplicationStateQueryResponse(ReplicationState state, String instanceName) {
        /* no waiting queue here
        LinkedBlockingQueue aQueue =
                ReplicationResponseRepository.putEmptyQueueEntry(state);
         */
        //send message over propagated pipe
        sendOverPropagatedPipe(state, instanceName, true);
        
        return state;
    }
    
    private Message createBroadcastMessage(ReplicationState state, boolean isResponse, String instanceName) {
        return ReplicationState.createBroadcastMessage(state, isResponse, instanceName);
    }     
    
    private boolean doesListContainAckRequiredState(List stateList) {
        boolean result = false;
        for(int i=0; i<stateList.size(); i++) {
            ReplicationState nextState = (ReplicationState)stateList.get(i);
            if(nextState.isAckRequired()) {
                result = true;
            }
            if(result) {
                break;
            }
        }
        return result;
    }    
    
    private static void displayList(Logger logger, List aList, String listName) {
        if (logger.isLoggable(Level.FINE)) {
            for (int i = 0; i < aList.size(); i++) {
                logger.log(Level.FINE, listName + " " + aList.get(i));
            }
        }
    }    
    
    private static void displayList(List aList, String listName) {
        displayList(_logger, aList, listName);
    } 
    
    private OutputPipe createPropagatedOutputPipe(String instanceName) {
        JxtaStarter jxtaStarter = JxtaStarter.createInstance();
        PeerGroup netPeerGroup = jxtaStarter.getNetPeerGroup();
        PipeService pipeService = netPeerGroup.getPipeService();
        PipeAdvertisement pipeAdv = JxtaUtil.getPropagatedPipeAdvertisement();
        //System.out.println("prop pipe adv: " + pipeAdv);
        OutputPipe op = null;
        try {
            op = pipeService.createOutputPipe(pipeAdv, Collections.singleton(JxtaStarter.getPeerID(instanceName)), 10000); 
            //op = pipeService.createOutputPipe(pipeAdv, 100);
        } catch (IOException ex) {
            if(_pipelogger.isLoggable(Level.INFO)) {
                _pipelogger.log(Level.INFO, "Exception creating propagated pipe", ex);
            } 
        }
        return op;
    }
    
    //this method does not have to be synchronized
    private boolean timeThresholdExceeded(long thresholdDuration, long previousTime) {
        if((System.currentTimeMillis() - previousTime) > thresholdDuration) {
            return true;
        } else {
            return false;
        }        
    }    
    
    //+++++++++++++++++++ Extra Helper Classes ++++++++++++++++++++++++++++++
    
    /**
     * Our thread factory that adds the threads to our thread group and names
     * the thread to something recognizable.
     */
    static class ReplicationSenderThreadFactory implements ThreadFactory {

        final AtomicInteger threadNumber = new AtomicInteger(1);
        final String name;
        final ThreadGroup threadgroup;

        ReplicationSenderThreadFactory(String name, ThreadGroup threadgroup) {
            this.name = name;
            this.threadgroup = threadgroup;
        }

        public Thread newThread(Runnable runnable) {
            Thread thread = new Thread(threadgroup, runnable, name + " - " + threadNumber.getAndIncrement(), 0);
            if (thread.isDaemon()) {
                thread.setDaemon(false);
            }
            if (thread.getPriority() != Thread.NORM_PRIORITY) {
                thread.setPriority(Thread.NORM_PRIORITY);
            }
            return thread;
        }
    }
    
    private class BulkMessageSender implements Runnable {

        private List list = null;
        private boolean needBulkAck = false;
        private RouteAdvertisement routeAdv = null;
        
        public BulkMessageSender(List list, boolean needBulkAck, RouteAdvertisement routeAdv) {
            this.list = list;
            this.needBulkAck = needBulkAck;
            this.routeAdv = routeAdv;
        }
        
        public void run() {
            createMessageAndSend(list, needBulkAck, routeAdv);
            list.clear();
        }
    } 
    
    private class DispatchThreadNoDupsAllowed implements Runnable {
        
        private volatile boolean done = false;
       
        private Thread thread;
        
        private LinkedBlockingQueue<Object> queue;
        
        public DispatchThreadNoDupsAllowed() {
            this.queue = new LinkedBlockingQueue<Object>();
            this.thread = new Thread(this);
            this.thread.setDaemon(true);
            thread.start();
        }
        
        public void wakeup() {
            synchronized(dispatchThreadNoDupsAllowedMonitorObject) {
                dispatchThreadNoDupsAllowedMonitorObject.notify();
            }            
            //queue.add(new Object());
        }
        
        public void run() {
            ReplicationHealthChecker.incrementDispatchThreadCount();
            boolean hasSignaled = false;
            while (! done) {
                try {
                    synchronized(dispatchThreadNoDupsAllowedMonitorObject) {
                        if(((Map)currentMapNoDupsAllowed.get()).size() < 
                                NUMBER_OF_SESSIONS_PER_MESSAGE) {
                            dispatchThreadNoDupsAllowedMonitorObject.wait(50L);
                        }
                    }
                    //Object ignorableToken = queue.take();
                    //System.out.println("no dups emerging from wait...");                    
                    long lastChangeTimeNoDups = lastChangeTimeNoDupsAllowed.get();
                    if(!ReplicationHealthChecker.isFlushThreadWaiting()) {
                        //normal processing
                        if ( ((Map)currentMapNoDupsAllowed.get()).size() > 0) {                     
                            flushAllMessagesFromCurrentMapNoDupsAllowed(false);
                        }
                    } else { //only if flushThreadWaiting
                        //flush once more if you have data
                        try {
                            if(((Map)currentMapNoDupsAllowed.get()).size() > 0) {
                                flushAllMessagesFromCurrentMapNoDupsAllowed(false);
                            }
                        } finally {
                            //signal you are finished unloading if you have not previously done so
                            if(!hasSignaled) {                            
                                CountDownLatch doneSignal = ReplicationHealthChecker.getDoneSignal();
                                doneSignal.countDown();
                                hasSignaled = true;
                            }
                        }
                    }                            
                } catch (InterruptedException inEx) {
                    this.done = true;
                } catch (Throwable t) {
                    _pipelogger.log(Level.FINE, "nodup thread exception:", t);
                }   finally {
                    //if done is set signal you are finished if you have not previously done so
                    if(done && !hasSignaled) {
                        CountDownLatch doneSignal = ReplicationHealthChecker.getDoneSignal();
                        doneSignal.countDown();
                        hasSignaled = true;
                    }
                }
            }
        }

    }  
    
    //Start unicast code
    
    private static final String NAMESPACE = ReplicationState.NAMESPACE;
    private static final String ROUTEADV = ReplicationState.ROUTEADV;
    private MessageTransport endpointRouter = null;
    private RouteControl routeControl = null;
    private MessageElement routeAdvElement = null;
    private PeerGroup netPeerGroup = null;
    
    void initializeRouteAdvElement() {
        //used to ensure up to date routes are used
        netPeerGroup = getNetPeerGroup();
        endpointRouter = (netPeerGroup.getEndpointService()).getMessageTransport("jxta");
        if (endpointRouter != null) {
            routeControl = (RouteControl) endpointRouter.transportControl(EndpointRouter.GET_ROUTE_CONTROL, null);
            RouteAdvertisement route = routeControl.getMyLocalRoute();
            if (route != null) {
                routeAdvElement = new TextDocumentMessageElement(ROUTEADV,
                        (XMLDocument) route.getDocument(MimeMediaType.XMLUTF8), null);
            }
        } 
    }
    
    PeerGroup getNetPeerGroup() {
        if(netPeerGroup == null) {
            JxtaStarter jxtaStarter = JxtaStarter.createInstance();
            netPeerGroup = jxtaStarter.getNetPeerGroup();
        }
        return netPeerGroup;
    }
    
    void addRoute(Message msg) {
        if (routeAdvElement != null && routeControl != null) {
            msg.addMessageElement(NAMESPACE, routeAdvElement);
        }
    }
    
    //Note: later this should be simplified
    //because each instance of this class only sends messages
    //to a single instance so this is overkill
    private static class PropagatedPipeCache {      
        static class PropagatedPipeWrapper {
            final OutputPipe pipe;
            final RouteAdvertisement routeAdv;
            
            private static final Logger _pipelogger = ReplicationUtil.getPipeLogger();

            PropagatedPipeWrapper(String instanceName, RouteAdvertisement routeAdv) throws IOException {
                JxtaStarter jxtaStarter = JxtaStarter.createInstance();
                PeerGroup netPeerGroup = jxtaStarter.getNetPeerGroup();
                PipeService pipeService = netPeerGroup.getPipeService();
                PipeAdvertisement pipeAdv = JxtaUtil.getPropagatedPipeAdvertisement();
                PeerID peerID = JxtaStarter.getPeerID(instanceName);
                pipe = new BlockingWireOutputPipe(netPeerGroup, pipeAdv, peerID, routeAdv);
                this.routeAdv = routeAdv;
                if (_pipelogger.isLoggable(Level.FINE)) {
                    _pipelogger.fine("create PropagatedPipe instance=" + instanceName + " peerId="+ peerID + " routeAdv=" + routeAdv);
                }
            }
            
            void close() {
                if (pipe != null) {
                    pipe.close();
                }
            }
        }
        
        static class PropagatedPipeWrapperPool {
            final int NUMBER_OF_PIPES = ReplicationUtil.getNumberOfOutputPipes();
            final List<PropagatedPipeWrapper> pipeWrapperArray 
                = new ArrayList<PropagatedPipeWrapper>(NUMBER_OF_PIPES);
            Object[] monitor = new Object[NUMBER_OF_PIPES];
            
            // index into pipeWrapperArray. value from range 0 .. NUMBER_OF_PIPES - 1
            final AtomicInteger nextIndex = new AtomicInteger(0);   
           
            private static final Logger _pipelogger = ReplicationUtil.getPipeLogger();
            
            PropagatedPipeWrapperPool() {
                initPipeWrapperArray();
                for (int i = 0; i < NUMBER_OF_PIPES; i++) {
                    monitor[i] = new Object();
                }
            }
            
            void initPipeWrapperArray() {            
                for(int i=0; i < NUMBER_OF_PIPES; i++) {
                    pipeWrapperArray.add(null);
                }
            }
            
            OutputPipe getNextPipe(String instanceName, RouteAdvertisement ra) throws IOException {
                int currentIndex;
                synchronized(nextIndex) {
                    currentIndex = nextIndex.getAndIncrement() % NUMBER_OF_PIPES;
                    nextIndex.compareAndSet(NUMBER_OF_PIPES, 0);
                }
                
                PropagatedPipeWrapper pipeWrapper = null;
                synchronized(monitor[currentIndex]) {
                    pipeWrapper = pipeWrapperArray.get(currentIndex);
                    if (pipeWrapper == null || pipeWrapper.pipe == null || pipeWrapper.pipe.isClosed() ){ 
                        pipeWrapper = new PropagatedPipeWrapper(instanceName, ra);
                        pipeWrapperArray.set(currentIndex, pipeWrapper);
                    } else if ( ! ra.equals(pipeWrapper.routeAdv)) { 
                         if (_pipelogger.isLoggable(Level.FINER)) {
                             _pipelogger.finer("detected a stale propagated pipe for instance=" + instanceName +
                                       " cached routeAdv=" + pipeWrapper.routeAdv + " current routeAdv=" + ra);
                         }
                         // close stale pipe
                         pipeWrapper.close();
                         pipeWrapper = new PropagatedPipeWrapper(instanceName, ra);
                         pipeWrapperArray.set(currentIndex, pipeWrapper);
                    } else if (_pipelogger.isLoggable(Level.FINEST)) {
                        _pipelogger.finest("cache hit for propagated pipe to instance=" + instanceName + " routeAdv=" + ra);    
                    }
                }
                return pipeWrapper.pipe;
            }
      
            void cleanout() {
                for (int i = 0; i < NUMBER_OF_PIPES; i++) {
                    synchronized (monitor[i]) {
                        PropagatedPipeWrapper pw = pipeWrapperArray.get(i);
                        if (pw != null) {
                            pw.close();
                        }
                        pipeWrapperArray.set(i, null);
                    }
                }
            }
        }        
        
        private static final Logger _pipelogger = ReplicationUtil.getPipeLogger();
        
        private ConcurrentHashMap<String, PropagatedPipeWrapperPool> cachedPropagatedPipes =
                        new ConcurrentHashMap<String, PropagatedPipeWrapperPool>();
        
        /**
         * get a propagated pipe matching instanceName AND routeAdvertisement.
         * Check cache first.
         * 
         * @param instanceName destination for unicast
         * @param ra           route advertisement sent from destination.
         * @return
         */
        public OutputPipe getPipe(String instanceName, RouteAdvertisement ra) throws IOException {
            PropagatedPipeWrapperPool pipeWrapperPool = null;
            
            synchronized(cachedPropagatedPipes) {
                pipeWrapperPool = cachedPropagatedPipes.get(instanceName);
                if (pipeWrapperPool == null) {
                    pipeWrapperPool = new PropagatedPipeWrapperPool();
                    cachedPropagatedPipes.put(instanceName, pipeWrapperPool);
                }
            }
            return pipeWrapperPool.getNextPipe(instanceName, ra);
        }     
        
        public void removePipe(String instanceName) {
            PropagatedPipeWrapperPool pool = null;
            synchronized(cachedPropagatedPipes) {
                 pool = cachedPropagatedPipes.get(instanceName);
            }
            if (pool != null) {
                pool.cleanout();
            }
            if (_pipelogger.isLoggable(Level.FINE)) {
                _pipelogger.fine("remove all cached propagagated pipe to instance=" + instanceName);
            }
        }
    }
    
    private PropagatedPipeCache propagatedPipeCache = new PropagatedPipeCache();    
    
    void removeCachedPropagatedOutputPipe(String instanceName) {
        propagatedPipeCache.removePipe(instanceName);
    }
    
    public void sendOverPropagatedPipe(ReplicationState state, String instanceName, boolean isResponse) {
        final int MAX_RETRY = 6;
        OutputPipe outputPipe = null;
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("JxtaReplicationUnicastSender>>sendOverPropagatedPipe:toInstance=" + instanceName);
        }        
        //System.out.println("JxtaReplicationUnicastSender>>sendOverPropagatedPipe:toInstance=" + instanceName);         
        //will use this to create our outputPipe
        RouteAdvertisement routeAdv = state.getRouteAdvertisement();
        
        boolean sendResult = false;
        for (int retryCount =0; retryCount < MAX_RETRY; retryCount++) {
            outputPipe = getPropagatedOutputPipe(instanceName, routeAdv);
            if(outputPipe == null) {
                continue;
            }
            Message theMsg = this.createBroadcastMessage(state, isResponse, instanceName);
            //Message theMsg = this.createMessage(state, isResponse);
            //temporarily try multicast -ok this worked
            //this.sendOverPropagatedPipe(theMsg);

            try {
                if(_pipelogger.isLoggable(Level.FINEST)) {
                    _pipelogger.log(Level.FINEST, "sending over propagated pipe for sessionid="+ state.getId() + " pipe=" + outputPipe.toString() +
                                    " to instance " + instanceName  );
                }
                
                sendResult = outputPipe.send(theMsg);           
                if (sendResult) {
                    break;
                } else {
                    if (_pipelogger.isLoggable(Level.FINE) ) {
                         _pipelogger.fine("send over propagated pipe failed for sessionid="+ state.getId() + " pipe=" + outputPipe.toString() +
                                    " to instance " + instanceName + " retryCount=" + retryCount);
                    }   
                }
            } catch (IOException ex) {
                if(_pipelogger.isLoggable(Level.FINE)) {
                    _pipelogger.log(Level.FINE, "IOException sending unicast message", ex);
                }
                removeCachedPropagatedOutputPipe(instanceName);
                if(retryCount < MAX_RETRY) {
                    if(_pipelogger.isLoggable(Level.FINE)) {
                        _pipelogger.fine("propagated unicast pipe send retrying");
                    }
                }
            }
        }
        if (sendResult == false) {
            _unicastMessageFailures.incrementAndGet();
            if (_pipelogger.isLoggable(Level.INFO)) {
                _pipelogger.info("failed to send message over unicast pipe for sessionid=" + state.getId() + " cmd="+ state.getCommand() + " to instance " +
                        instanceName + " pipe=" + outputPipe);
            }
        }
    } 
    
    private OutputPipe getPropagatedOutputPipe(String instanceName, RouteAdvertisement routeAdv) {
        //if routeAdv is null use instance name based approach
        if(routeAdv == null) {
            return createPropagatedOutputPipe(instanceName);            
        }
        OutputPipe result = null;
        try {
            result = propagatedPipeCache.getPipe(instanceName, routeAdv);
        } catch (IOException ex) {
            if (_pipelogger.isLoggable(Level.INFO)) {
                _pipelogger.log(Level.INFO, "JxtaReplicationUnicastSender::getPropagatedOutputPipe got IOException "
                        + instanceName + " pipe= " + result, ex);
            }
        }
        return result;
    }    
    
    //End unicast code  

}
