/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.enterprise.ee.web.sessmgmt;

import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import com.sun.enterprise.web.ServerConfigLookup;

/**
 * Provide ability to monitor and control using data on demand
 * loads
 * @author Larry White
 */
public class LoadProcessingGovernor {
    
    public final static String LOGGER_MEM_REP 
        = ReplicationState.LOGGER_MEM_REP;    
    
    /**
     * The logger to use for logging ALL pipe related messages.
     */     
    private static final Logger _pipelogger = ReplicationUtil.getPipeLogger();
    private static final Logger _logger = LogDomains.getLogger(LogDomains.WEB_LOGGER);

    private static final AtomicInteger _currentLoads = new AtomicInteger(0);
    private static final AtomicInteger _previousLoads = new AtomicInteger(0);
    private static int LOAD_THRESHOLD = 200;
    
    private static final AtomicLong _lastStartTime = new AtomicLong(-1L);
    private static final AtomicLong _lastEndTime 
        = new AtomicLong(System.currentTimeMillis());
    private static final AtomicLong _lastStressDuration = new AtomicLong(-1L);
    private static final AtomicBoolean _wasEverStressed = new AtomicBoolean(false);
    private static final AtomicBoolean _stressDetected = new AtomicBoolean(false);
    private static final AtomicBoolean _onStressEndCompleted = new AtomicBoolean(true);
    
    
    private static boolean _replicationDemandLoadTuning = true;
    
    private TuningDebugger tuningDebuggerThread = null;
    
    /**
     * The singleton instance of LoadProcessingGovernor
     */    
    private final static LoadProcessingGovernor _soleInstance 
        = new LoadProcessingGovernor();    
    
    /** Return the singleton instance
     *  returns the sole instance of LoadProcessingGovernor
     */
    public static LoadProcessingGovernor getInstance() {
        return _soleInstance;
    }    
   
    /** Creates a new instance of LoadProcessingGovernor */
    private LoadProcessingGovernor() {
        initialize();
    }
    
    private void initialize() {
        ServerConfigLookup lookup = new ServerConfigLookup();
        _replicationDemandLoadTuning = lookup.isReplicationLoadGovernorEnabled();
        LOAD_THRESHOLD = lookup.getReplicationLoadGovernorThresholdFromConfig();
        if(_pipelogger.isLoggable(Level.FINE)) {
            _pipelogger.fine("_replicationDemandLoadTuning = " + _replicationDemandLoadTuning);
            _pipelogger.fine("LOAD_THRESHOLD = " + LOAD_THRESHOLD);
        }
        if(_replicationDemandLoadTuning) {
            tuningDebuggerThread = new TuningDebugger();
        }
    }
    
    public static boolean isSystemUnderLoadStress() {           
        return ((getCurrentLoads() > LOAD_THRESHOLD) || (getPreviousLoads() > LOAD_THRESHOLD));
    }    
        
    public static void incrementCurrentLoads() {
        if(!isLoadGoverningEnabled()) {
            return;
        } 
        _currentLoads.incrementAndGet();               
        if(isSystemUnderLoadStress()) {
            if(_stressDetected.compareAndSet(false, true)) {
                onStressBegins();
            }
        }
    }    
    
    public static int getCurrentLoads() {
        return _currentLoads.get();
    }
    
    public static int getPreviousLoads() {
        return _previousLoads.get();
    }    
    
    public static void reset() {
        if(!isSystemUnderLoadStress() && _stressDetected.get()) {
            if(_onStressEndCompleted.compareAndSet(false, true)) {
                onStressEnds();
            } 
        }
        _previousLoads.set(getCurrentLoads());
        _currentLoads.set(0);        
    }
    
    public static long getLastStartTime() {
        return _lastStartTime.get();
    }
    
    public static long getLastEndTime() {
        return _lastEndTime.get();
    }
    
    public static long getLastStressDuration() {
        return _lastStressDuration.get();
    }
    
    public static void onStressBegins() {
        _logger.log(Level.INFO, "LoadProcessingGovernor>>onStressBegins");
        _lastStartTime.set(System.currentTimeMillis());
        _onStressEndCompleted.set(false);
    }
    
    public static void onStressEnds() {
        _logger.log(Level.INFO, "LoadProcessingGovernor>>onStressEnds");
        _lastEndTime.set(System.currentTimeMillis());                
        _lastStressDuration.set(getLastEndTime() - getLastStartTime());
        _wasEverStressed.set(true);
        _stressDetected.set(false);
    } 
    
    public static boolean isBeyondExtraDuration() {
        if(!isLoadGoverningEnabled() || !_wasEverStressed.get()) {
            return true;
        }         
        if(isSystemUnderLoadStress()) {
            return false;
        }
        return( (System.currentTimeMillis() - getLastEndTime()) > 2 * getLastStressDuration());
    }
    
    private static boolean isLoadGoverningEnabled() {
        return _replicationDemandLoadTuning;
    }
    
    private class TuningDebugger implements Runnable {
        
        private Thread thread;
        private boolean threadDone = false;
        private int SLEEP_INTERVAL_SECONDS = 15; 
        
        public TuningDebugger() {
            this.thread = new Thread(this);
            this.thread.setDaemon(true);
            thread.start();            
        }
        
        public void run() {
            try { 
		// Loop until the termination semaphore is set
		while (!threadDone) {
		    threadSleep();
		    doTuningDebug();
		}
	    } catch (Throwable t) {
		_logger.log(Level.WARNING, "LoadProcessingGovernor.TuningDebugger thread terminated, here is stack trace", t);
	    } finally {
	        _logger.log(Level.INFO, "LoadProcessingGovernor.TuningDebugger thread completed.");
	    }
        }        
        
        private void doTuningDebug() {
            if(isLoadGoverningEnabled()) {           
                if (_pipelogger.isLoggable(Level.FINE)) {
                    _pipelogger.fine("LoadProcessingGovernor:_currentLoads = " + getCurrentLoads());
                    _pipelogger.fine("LoadProcessingGovernor_previousLoads = " + getPreviousLoads());
                    _pipelogger.fine("_stressDetected.get() = " + _stressDetected.get());
                    _pipelogger.fine("_onStressEndCompleted.get() = " + _onStressEndCompleted.get());
                    _pipelogger.fine("_wasEverStressed.get() = " + _wasEverStressed.get());
                    _pipelogger.fine("getLastStressDuration() = " + getLastStressDuration());
                    _pipelogger.fine("isBeyondExtraDuration() = " + isBeyondExtraDuration());
                    _pipelogger.fine("getLastStartTime() = " + getLastStartTime());
                    _pipelogger.fine("getLastEndTime() = " + getLastEndTime());
                    _pipelogger.fine("isSystemUnderLoadStress() = " + isSystemUnderLoadStress());
                }
            }
            reset();
        }
        
        /**
         * Sleep for SLEEP_INTERVAL_SECONDS seconds
         */
        private void threadSleep() {
            try {
                Thread.sleep(SLEEP_INTERVAL_SECONDS * 1000L);            
            } catch (InterruptedException e) {
                ;
            }
        }        
                 
    }    
}
