/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
/*
/*
 * PlannedShutdownNotificationEventHandler.java
 *
 * Created on November 30, 2006, 11:39 AM
 * This class processes planned shutdown notifications from GMS
 *
 */

package com.sun.enterprise.ee.web.sessmgmt;

import com.sun.enterprise.ee.cms.core.*;
import com.sun.logging.LogDomains;
import java.util.logging.Logger;
import java.util.logging.Level;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collection;
import java.util.concurrent.atomic.AtomicBoolean;

import com.sun.enterprise.web.ServerConfigLookup;

/**
 *
 * @author lwhite
 */
public class PlannedShutdownNotificationEventHandler implements CallBack {
    
    private static HashMap pendingShutdowns = new HashMap();
    private static AtomicBoolean processingShutdownFlag 
        = new AtomicBoolean(false);    
    
    private static final Logger _logger = LogDomains.getLogger(LogDomains.WEB_LOGGER);
    
    /** Creates a new instance of PlannedShutdownNotificationEventHandler */
    public PlannedShutdownNotificationEventHandler() {
    }
    
    public void processNotification(Signal notification) {        
        String failedPartnerInstance = notification.getMemberToken();
        if (_logger.isLoggable(Level.FINE)) {
            _logger.fine("Received Planned Shutdown Notification: " + failedPartnerInstance);
        }
        if(notification instanceof PlannedShutdownSignal){
            PlannedShutdownSignal plannedShutdownSignal 
                = (PlannedShutdownSignal)notification;
            if(plannedShutdownSignal.getEventSubType().equals(
                GMSConstants.shutdownType.GROUP_SHUTDOWN)) {
                if (_logger.isLoggable(Level.FINE)) {
                    _logger.fine("cluster shutting down:set stopping true");
                }            
                ReplicationHealthChecker.setStopping(true);
                return;
            }
            ReplicationHealthChecker healthChecker
                = ReplicationHealthChecker.getInstance();
            //process immediately except where it is our
            //replicate from partner - this case will be handled
            //as pending
            if(!healthChecker.isReplicateFromPartner(failedPartnerInstance)) {
                doProcessShutdown(failedPartnerInstance);
            } else {
                addPendingShutdown(failedPartnerInstance);
            }            
        }         
    }
    
    static void doProcessShutdown(String failedPartnerInstance) { 
        if(!ReplicationHealthChecker.isStopping()) {            
            JxtaReplicationReceiver jxtaReplicationReceiver
                = (JxtaReplicationReceiver) ReplicationHealthChecker.getReplicationReceiver();
            if (_logger.isLoggable(Level.FINE)) {
                _logger.fine("planned shutdown notification causing call to respondToFailure");
            }
            jxtaReplicationReceiver.respondToFailure(failedPartnerInstance, false);
            ReplicationHealthChecker healthChecker
                = ReplicationHealthChecker.getInstance();
            healthChecker.removeIfExtraInstance(failedPartnerInstance); 
            ReplicationMessageRouter router
                = ReplicationMessageRouter.createInstance();
            router.handleDynamicOwnershipChanges();                                 
        }
    }
    
    static void addPendingShutdown(String instanceName) {
        PendingShutdown pendingShutdown = new PendingShutdown(instanceName, System.currentTimeMillis());
        pendingShutdowns.put(instanceName, pendingShutdown);
    }
    
    static void removePendingShutdown(String instanceName) {
        pendingShutdowns.remove(instanceName);
    }
    
    static boolean isShutdownPending(String instanceName) {
        long timeNow = System.currentTimeMillis();
        PendingShutdown pendingShutdown = (PendingShutdown)pendingShutdowns.get((String)instanceName);
        if(pendingShutdown == null) {
            return false;
        }
        long potentialUnloadTime = getShutdownWaitTimeMillis();
        long pendingShutdownTimeMillis = pendingShutdown.getCreationTime();
        //pending shutdown is marked valid or pending for less than 5 min
        if(pendingShutdown.isValid() || (timeNow - pendingShutdownTimeMillis) < potentialUnloadTime ) {
            if (_logger.isLoggable(Level.FINE)) {
                _logger.fine("shutdown is pending");
            }
            //set as valid pendingShutdown in case it is not processed
            //right away
            pendingShutdown.setValid(true);
            return true;
        } else {
            //pending shutdown is not marked valid and more than 5 min old
            //so it should be removed
            if (_logger.isLoggable(Level.FINE)) {
                _logger.fine("shutdown is not pending or more than 5 min old");
            }
            removePendingShutdown(instanceName);
            return false;
        }
    }
    
    private static long getShutdownWaitTimeMillis() {
        int maxSessionUnloadTimeSeconds 
            = getMaxSessionUnloadTimeInSeconds();
        if(maxSessionUnloadTimeSeconds < 5000) {
            return 5000L;
        } else {
            return maxSessionUnloadTimeSeconds * 1000L;
        }
    }
    
    private static int getMaxSessionUnloadTimeInSeconds() {
        ServerConfigLookup lookup = new ServerConfigLookup();
        return lookup.getMaxSessionUnloadTimeInSecondsPropertyFromConfig();
    }    
    
    static void checkAndDoShutdownFor(String partnerInstance) {
        //do not bother if stopping
        if(ReplicationHealthChecker.isStopping()) {
            return;
        }        
        //do not bother if no pending shutdown for this proposed instance
        //this could include a pending shutdown timing out after 5 min.
        if(!isShutdownPending(partnerInstance)) {
            //FIXME next line is work-around for gms shutdown notifications
            //failing to arrive consistently
            //so if a pipeCloseEvent triggers the checkAndDoShutdownFor and we 
            //do not yet have a pending shutdown, rather than quit we
            //add the pending shutdown ourselves here and try to go ahead
            //addPendingShutdown(newPartnerInstance);
            //next line had this return statement temporarily commented out
            return;
            //end FIXME
        }
        if(reserveProcessingShutdown()) {
            if (_logger.isLoggable(Level.FINE)) {
                _logger.fine("got reservation:entering doProcessShutdown:instanceName:" + partnerInstance);
            }
            //we have successfully reserved shutdown processing
            // - go ahead and process
            doProcessShutdown(partnerInstance);
        } else {
            if (_logger.isLoggable(Level.FINE)) {
                _logger.fine("refused reservation:entering doPollCheckForShutdownProcessing");
            }
            //must enter a poll wait
            doPollCheckForShutdownProcessing(1000L);
        }
    } 
    
    private static void doPollCheckForShutdownProcessing(long sleepTime) {
        //FIXME might want a timeout for this
        if (_logger.isLoggable(Level.FINER)) {
            _logger.entering("PlannedShutdownNotificationEventHandler", "doPollCheckForShutdownProcessing", sleepTime);
        }
        boolean shouldContinue = true;
        while(shouldContinue) {
            try {
                Thread.currentThread().wait(sleepTime);
            } catch (InterruptedException ex) {
                //deliberate no-op
            }
            PendingShutdown nextPendingShutdown = getNextPendingShutdown();
            if(nextPendingShutdown == null) {
                shouldContinue = false;
                break;
            }
            String nextPartnerInstance = nextPendingShutdown.getInstanceName();
            checkAndDoShutdownFor(nextPartnerInstance);
        }
    }
        
    private static PendingShutdown getNextPendingShutdown() {
        //get next pending shutdown - earliest timestamp
        PendingShutdown result = null;
        ArrayList pendingShutdownsToBeRemoved = new ArrayList();
        Collection pendShutdowns = pendingShutdowns.values();
        Iterator it = pendShutdowns.iterator();
        //while iterating also collect and then remove any expired instanceNames
        while(it.hasNext()) {
            PendingShutdown nextPendingShutdown = (PendingShutdown)it.next();
            //skip and add to remove list any expired instanceName
            if(nextPendingShutdown.isCandidateForRemoval(30000L)) {
                pendingShutdownsToBeRemoved.add(nextPendingShutdown.getInstanceName());
                continue;
            }
            if(result == null) {
                result = nextPendingShutdown;
            } else {
                if(nextPendingShutdown.getCreationTime() < result.getCreationTime()) {
                    result = nextPendingShutdown;
                }
            }
        }
        removeAll(pendingShutdownsToBeRemoved);
        return result;
    }        
        
    private static void removeAll(ArrayList removeList) {
        for(int i=0; i<removeList.size(); i++) {
            String nextInstanceName = (String)removeList.get(i);
            removePendingShutdown(nextInstanceName);
        }
    }
    
    public static boolean isProcessingShutdown() {
        return processingShutdownFlag.get();
    }
    
    /** attempt to set processing to true if it is false
     * else it fails
     */
    public static boolean reserveProcessingShutdown() {
        return (processingShutdownFlag.compareAndSet(false, true));
    }    
    
    public static void setProcessingShutdown(boolean value) {
        processingShutdownFlag.set(value);
    }   
    
}
