/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.cli.commands;

import java.io.File;
import java.lang.reflect.Method;
import java.io.PrintStream;
import java.io.FileOutputStream;
import java.io.IOException;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.cli.framework.CommandException;

/**
 *  This class uses Java reflection to invoke Derby
 *  NetworkServerControl class.
 *  This class is used to start/stop/ping derby database.
 *  The reason for creating this class instead of directly
 *  invoking NetworkServerControl from the StartDatabaseCommand
 *  class is so that a separate JVM is launched when starting the
 *  database and the control is return to CLI.  
 *  @author <a href="mailto:jane.young@sun.com">Jane Young</a>
 *  @version  $Revision: 1.12 $
 */
public final class DerbyControl
{
    final public static String DB_LOG_FILENAME = "derby.log";
    final private String derbyCommand;
    final private String derbyHost;
    final private String derbyPort;
    final private String derbyHome;
    final private boolean redirect;

        //constructor 
    public DerbyControl(final String dc, final String dht, final String dp,
                        final String redirect, final String dhe)
    {
        this.derbyCommand = dc;
        this.derbyHost = dht;
        this.derbyPort = dp;
        this.derbyHome = dhe;
	    this.redirect = Boolean.valueOf(redirect).booleanValue();
	    if (this.redirect) {

	        try {
                String dbLog = "";
                if (this.derbyHome == null) {
                    // if derbyHome is null then redirect the output to a temporary file
                    // which gets deleted after the jvm exists.
		            dbLog = createTempLogFile();
                }
		    else {
		        dbLog = createDBLog(this.derbyHome);
		    }
            
                //redirect stdout and stderr to a file
                PrintStream printStream = new PrintStream(new FileOutputStream(dbLog, true), true);
                System.setOut(printStream);
                System.setErr(printStream);
            }
            catch (Throwable t) {
	            t.printStackTrace();
	            //exit with an error code of 2
	            System.exit(2);
            }
        }
        //do not set derby.system.home if derbyHome is empty
        if (this.derbyHome!=null && this.derbyHome.length()>0) {
            System.setProperty("derby.system.home", this.derbyHome);
        }
	    //set the property to not overwrite log file
	    System.setProperty("derby.infolog.append", "true");
    }
    
        //constructor
    public DerbyControl(final String dc, final String dht, final String dp)
    {
        this(dc,dht,dp,"true", null);
    }
    
        //constructor
    public DerbyControl(final String dc, final String dht, final String dp, final String redirect)
    {
        this(dc,dht,dp,redirect,null);
    }
    
        /**
         * This methos invokes the Derby's NetworkServerControl to start/stop/ping
         * the database.
         */
    private void invokeNetworkServerControl()
    {
        try {
            Class networkServer = Class.forName("org.apache.derby.drda.NetworkServerControl");
            Method networkServerMethod = networkServer.getDeclaredMethod("main",
                                                       new Class[]{String[].class});
            Object [] paramObj = new Object[]{new String[]{derbyCommand, "-h", derbyHost, "-p", derbyPort}};
           
            networkServerMethod.invoke(networkServer, paramObj);
        }
        catch (Throwable t) {
	        t.printStackTrace();
	        System.exit(2);
        }
    }


        /**
         * create a db.log file that stdout/stderr will redirect to.
         * dbhome is the derby.system.home directory where derb.log
         * gets created.  if user specified --dbhome options, derby.log
         * will be created there.
         **/
    private String createDBLog(final String dbHome) throws Exception
    {
        //dbHome must exist and  have write permission
        final File fDBHome = new File(dbHome);
	    String dbLogFileName = "";

        final StringManager lsm = StringManager.getManager(DerbyControl.class);
        if (fDBHome.isDirectory() && fDBHome.canWrite()) {
            final File fDBLog = new File(dbHome, DB_LOG_FILENAME);
	        dbLogFileName = fDBLog.toString();

            //if the file exists, check if it is writeable
            if (fDBLog.exists() && !fDBLog.canWrite()) {
	        System.out.println(lsm.getString("UnableToAccessDatabaseLog", dbLogFileName));
	        System.out.println(lsm.getString("ContinueStartingDatabase"));
	        //if exist but not able to write then create a temporary 
	        //log file and persist on starting the database
	        dbLogFileName = createTempLogFile();
            }
            else if (!fDBLog.exists()) {
                //create log file
                fDBLog.createNewFile();
            }
        }
        else {
            System.out.println(lsm.getString("InvalidDirectory", dbHome));
	        System.out.println(lsm.getString("ContinueStartingDatabase"));
	        //if directory does not exist then create a temporary log file
	        //and persist on starting the database
	        dbLogFileName = createTempLogFile();
        }
        return dbLogFileName;
    }

   /**
      creates a temporary log file.
   */
    private String  createTempLogFile() throws CommandException
    {
        String tempFileName = "";
        try {
            final File fTemp = File.createTempFile("foo", null);
            fTemp.deleteOnExit();
	        tempFileName = fTemp.toString();
	    }
	    catch (IOException ioe) {
            final StringManager lsm = StringManager.getManager(DerbyControl.class);
	        throw new CommandException(lsm.getString("UnableToAccessDatabaseLog", tempFileName));
	    }
        return tempFileName;
    }    
    
    public static void main(String[] args) {
        
        if (args.length<3){
            System.out.println("paramters not specified.");
            System.out.println("DerbyControl <derby command> <derby host> <derby port> <derby home> <redirect output>");
            System.exit(1);
        }
        
        DerbyControl derbyControl = null;
        if (args.length == 3)
            derbyControl = new DerbyControl(args[0], args[1], args[2]);
        else if (args.length == 4 )
            derbyControl = new DerbyControl(args[0], args[1], args[2], args[3]);
        else if (args.length > 4)
            derbyControl = new DerbyControl(args[0], args[1], args[2], args[3], args[4]);
        if (derbyControl != null)
            derbyControl.invokeNetworkServerControl();
    }
}


    
