/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.cli.commands.monitor;

import com.sun.enterprise.cli.framework.*;
import com.sun.appserv.management.monitor.*;
import com.sun.appserv.management.monitor.statistics.*;
import javax.management.j2ee.statistics.*;
import com.sun.appserv.management.j2ee.statistics.*;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.appserv.management.util.misc.MapUtil;
import java.util.StringTokenizer;
import java.util.Map;
import java.util.Timer;
import java.util.List;
import java.util.Vector;
import java.io.File;

public class EntityBeanMonitorTask extends BeanMonitorTask
{
    private final String displayFormat = "%1$-10s %2$-10s %3$-10s %4$-10s %5$-10s %6$-10s";

    public EntityBeanMonitorTask(final ServerRootMonitor srm, final String filter, final Timer timer,
                              final boolean verbose, final File fileName)
        throws MonitorTaskException
    {
        super(srm, filter, timer, verbose, fileName);
        
        if (this.filter == null)
            getDefaultFilter();
        else
        {
            final StringTokenizer st = new StringTokenizer(this.filter, ":");
            if (st.countTokens() < 2 )
                throw new MonitorTaskException("Invalid filter format.  EntityBeans are in either ejbmodule "+
                                               "or appmodule.  Please use the format: <ejbmodule>:<entitybean> " +
                                               "or <appmodule>:<ejbmodule>:<entitybean>.");
            else if (st.countTokens() == 2)
            {
                ejbName = st.nextToken();
                beanName = st.nextToken();
            }
            else {
                appName = st.nextToken();
                ejbName = st.nextToken();
                beanName = st.nextToken();
            }
            verifyFilterValue();
        }
        final String title = String.format("%1$50s", "Entity Bean Monitoring " + this.filter);
        System.out.println(title);
        displayHeader();        
    }
    

    public void run()
    {
        if (srm == null) {
            super.cancelMonitorTask();
            return;
        }
        
        Map<String,EJBModuleMonitor> ejbMap = null;
        if (appName == null)
        {
            ejbMap = srm.getEJBModuleMonitorMap();
        }
        else
        {
            final Map<String,ApplicationMonitor> appMap = srm.getApplicationMonitorMap();
            if (appMap == null || appMap.size()<1) {
                cancelMonitorTask();
                return;
            }
            final ApplicationMonitor am = appMap.get(appName);
            ejbMap = am.getEJBModuleMonitorMap();
        }
        if (ejbMap == null || ejbMap.size()<1) {
            cancelMonitorTask();
            return;
        }
        
        final EntityBeanMonitor entitybeanMonitor = getEntityBeanMonitor(ejbMap);
        if (entitybeanMonitor == null) {
            cancelMonitorTask();
            return;
        }
        
        final EntityBeanStats entitybeanStat = entitybeanMonitor.getEntityBeanStats();
        if (verbose && counter == NUM_ROWS)
        {
            displayHeader();
            counter = 0;  //reset to 0
        }
        displayData(entitybeanStat);
        if (verbose) counter++;
    }

    
    private EntityBeanMonitor getEntityBeanMonitor(final Map<String,EJBModuleMonitor> ejbMap)
    {
        final EJBModuleMonitor ejbModuleMonitor = ejbMap.get(ejbName);
        final Map<String,EntityBeanMonitor> entitybeanMap = ejbModuleMonitor.getEntityBeanMonitorMap();
        return entitybeanMap.get(beanName);
    }
    
    
    List<String> getBeansInEjbModule(final String appName,
                                     final Map<String,EJBModuleMonitor> ejbModuleMap)
    {
        List<String> possibleEntityBeans = new Vector<String>();
        final String[] ejbModules = MapUtil.getKeyStrings(ejbModuleMap);
        if (ejbModuleMap != null && ejbModuleMap.size() > 0)
        {
            for (String ejbModule : ejbModules)
            {
                final EJBModuleMonitor ejbModuleMonitor = ejbModuleMap.get(ejbModule);
                final Map<String,EntityBeanMonitor> entityBeanMap = ejbModuleMonitor.getEntityBeanMonitorMap();
                final String[] entityBeans = MapUtil.getKeyStrings(entityBeanMap);
                for (String entityBean : entityBeans)
                {
                    if (appName == null)
                        possibleEntityBeans.add(ejbModule+":"+entityBean);
                    else
                        possibleEntityBeans.add(appName+":"+ejbModule+":"+entityBean);
                }
            }
        }
        return possibleEntityBeans;
    }

    
        /**
         * returns true if entitybean exists in ejbmodule
         * @throws MonitorTaskException if ejbmodule is invalid.
         */
    boolean checkIfBeanExists(final Map<String,EJBModuleMonitor> ejbModuleMap)
        throws MonitorTaskException
    {
        if (!ejbModuleMap.containsKey(ejbName))
        {
            if (appName == null)
                throw new MonitorTaskException(ejbName + " does not exist.");
            else
                throw new MonitorTaskException(ejbName + " does not exist in " + appName+ ".");                
        }
        else {
            final EJBModuleMonitor ejbmoduleMonitor = ejbModuleMap.get(ejbName);
            final Map<String,EntityBeanMonitor> entityBeanMap = ejbmoduleMonitor.getEntityBeanMonitorMap();
            if (!entityBeanMap.containsKey(beanName))
                return false;
            else
                return true;
        }
    }
    
    
    private void displayHeader()
    {
        final String header = String.format("%1$20s %2$30s",
                                            "PooledCount", "ReadyCount");
        final String subHeader = String.format(displayFormat,
                                            "low","hi","current","low","hi","current");
        System.out.println(header);
        System.out.println(subHeader);        
        if (fileName != null)
        {
            writeToFile("PooledCount:low,PooledCount:hi,PooledCount:current,"+
                        "ReadyCount:low,ReadyCount:hi,ReadyCount:current");
        }
    }

    
    private void displayData(final EntityBeanStats stat)
    {
        final String data = String.format(displayFormat,
                                          stat.getPooledCount().getLowWaterMark(),
                                          stat.getPooledCount().getHighWaterMark(),
                                          stat.getPooledCount().getCurrent(),
                                          stat.getReadyCount().getLowWaterMark(),
                                          stat.getReadyCount().getHighWaterMark(),
                                          stat.getReadyCount().getCurrent());
        System.out.println(data);
        if (fileName != null)
        {
            final String fileData = String.format("%1$s,%2$s,%3$s,%4$s,%5$s,%6$s",
                                          stat.getPooledCount().getLowWaterMark(),
                                          stat.getPooledCount().getHighWaterMark(),
                                          stat.getPooledCount().getCurrent(),
                                          stat.getReadyCount().getLowWaterMark(),
                                          stat.getReadyCount().getHighWaterMark(),
                                          stat.getReadyCount().getCurrent());
            writeToFile(fileData);
        }
    }

    
    public void displayDetails()
    {
        System.out.println("********************************************************");
        System.out.println("* PooledCount = Number of beans in the pooled state.   *");
        System.out.println("* ReadyCount  = Number of beans in the ready state.    *");
        System.out.println("* low         = Low water mark                         *");
        System.out.println("* hi          = High water mark                        *");
        System.out.println("* current     = Current count                          *");
        System.out.println("********************************************************");
    }


}
    
    

