/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.cli.commands.monitor;

import com.sun.enterprise.cli.framework.*;
import com.sun.appserv.management.monitor.*;
import com.sun.appserv.management.monitor.statistics.*;
import javax.management.j2ee.statistics.*;
import com.sun.appserv.management.j2ee.statistics.*;
import javax.management.MBeanServerConnection;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.appserv.management.util.misc.MapUtil;
import java.util.Map;
import java.util.HashMap;
import java.util.TimerTask;
import java.util.Timer;
import java.io.File;

public class ThreadPoolMonitorTask extends MonitorTask
{
    private final String displayFormat = "%1$-5s %2$-5s %3$-5s %4$-5s %5$-5s %6$-5s %7$-5s "+
                                         "%8$-20s %9$-5s %10$-5s %11$-5s %12$-5s %13$-5s %14$-5s";

    public ThreadPoolMonitorTask(final ServerRootMonitor srm, final String filter, final Timer timer,
                                 final boolean verbose, final File fileName)
        throws MonitorTaskException
    {
        super(srm, filter, timer, verbose, fileName);
        final Map<String,ThreadPoolMonitor> threadpoolMap = srm.getThreadPoolMonitorMap();
        if (threadpoolMap == null)
            throw new MonitorTaskException("Unable to monitor threadpool.");
        final String[] keys = MapUtil.getKeyStrings(threadpoolMap);
        if (this.filter == null)
        {
            checkForNumberOfElements(keys);
            this.filter = keys[0];
        }
        else {
            if (!threadpoolMap.containsKey(this.filter)) {
                throw new MonitorTaskException(this.filter + " does not exist");
            }
        }
        final String title = String.format("%1$70s", "ThreadPool Monitoring: "+this.filter);
        System.out.println(title);
        displayHeader();        
    }
    

    public void run()
    {
        if (srm == null) {
            super.cancelMonitorTask();
            return;
        }
        
        final Map<String,ThreadPoolMonitor> threadpoolMap = srm.getThreadPoolMonitorMap();
        
            //maybe lost connection?
        if (threadpoolMap == null || threadpoolMap.size()<1) {
            cancelMonitorTask();
            return;
        }
        
        if (verbose && counter == NUM_ROWS)
        {
            displayHeader();
            counter = 0;  //reset to 0
        }
        monitorThreadPool(filter, threadpoolMap);
        if (verbose) counter++;
    }

    
    private void monitorThreadPool(final String element,
                                   final Map<String,ThreadPoolMonitor> threadpoolMap)
    {
        final ThreadPoolMonitor threadpoolMonitor = threadpoolMap.get(element);
        final ThreadPoolStats threadpoolStats = threadpoolMonitor.getThreadPoolStats();

        displayData(threadpoolStats);
    }

    
    private void displayHeader()
    {
        final String header = String.format("%1$2s %2$18s %3$30s",
                                            "AvgTimeInQueue", "AvgWorkCompTime",
                                            "CurrNumOfThreads");
        final String subHeader = String.format(displayFormat,
                                               "low","hi","cur","low","hi","cur","min","max",
                                               "low","hi","cur","avl","busy","add");
        System.out.println(header);
        System.out.println(subHeader);
        if (fileName != null)
        {
            writeToFile("AverageTimeInQueue:low,AverageTimeInQueue:hi,AverageTimeInQueue:current,"+
                        "AverageWorkCompletionTime:low,AverageWorkCompletionTime:hi,"+
                        "AverageWorkCompletionTime:current,AverageWorkCompletionTime:min,"+
                        "AverageWorkCompletionTime:max,CurrentNumberOfThreads:low,"+
                        "CurrentNumberOfThreads:hi,CurrentNumberOfThreads:current,"+
                        "NumberOfAvailableThreads,NumberOfBusyThreads,TotalWorkItemsAdded");
        }
        
    }

    
    private void displayData(final ThreadPoolStats tps)
    {
        final String data = String.format(displayFormat,
                                          tps.getAverageTimeInQueue().getLowWaterMark(),
                                          tps.getAverageTimeInQueue().getHighWaterMark(),
                                          tps.getAverageTimeInQueue().getCurrent(),
                                          tps.getAverageWorkCompletionTime().getLowWaterMark(), 
                                          tps.getAverageWorkCompletionTime().getHighWaterMark(),
                                          tps.getAverageWorkCompletionTime().getCurrent(),
                                          tps.getCurrentNumberOfThreads().getLowerBound(),
                                          tps.getCurrentNumberOfThreads().getUpperBound(),
                                          tps.getCurrentNumberOfThreads().getLowWaterMark(),
                                          tps.getCurrentNumberOfThreads().getHighWaterMark(),
                                          tps.getCurrentNumberOfThreads().getCurrent(),
                                          tps.getNumberOfAvailableThreads().getCount(),
                                          tps.getNumberOfBusyThreads().getCount(),
                                          tps.getTotalWorkItemsAdded().getCount());
        System.out.println(data);
        if (fileName != null)
        {
            final String fileData = String.format("%1$s,%2$s,%3$s,%4$s,%5$s,%6$s,%7$s,%8$s,%9$s,"+
                                                  "%10$s,%11$s,%12$s,%13$s,%14$s",
                                                  tps.getAverageTimeInQueue().getLowWaterMark(),
                                                  tps.getAverageTimeInQueue().getHighWaterMark(),
                                                  tps.getAverageTimeInQueue().getCurrent(),
                                                  tps.getAverageWorkCompletionTime().getLowWaterMark(), 
                                                  tps.getAverageWorkCompletionTime().getHighWaterMark(),
                                                  tps.getAverageWorkCompletionTime().getCurrent(),
                                                  tps.getCurrentNumberOfThreads().getLowerBound(),
                                                  tps.getCurrentNumberOfThreads().getUpperBound(),
                                                  tps.getCurrentNumberOfThreads().getLowWaterMark(),
                                                  tps.getCurrentNumberOfThreads().getHighWaterMark(),
                                                  tps.getCurrentNumberOfThreads().getCurrent(),
                                                  tps.getNumberOfAvailableThreads().getCount(),
                                                  tps.getNumberOfBusyThreads().getCount(),
                                                  tps.getTotalWorkItemsAdded().getCount());
            writeToFile(fileData);
        }
    }

    
    public void displayDetails()
    {
        System.out.println("*************************************************************************");
        System.out.println("* AvgTimeInQueue   = Average time a work item waits in the work queue   *");
        System.out.println("* AvgWorkCompTime  = Average elapsed time taken to complete a work item *");
        System.out.println("* CurrNumOfThreads = Current number of total threads                    *");
        System.out.println("* low  = Low water mark                                                 *");
        System.out.println("* hi   = High water mark                                                *");
        System.out.println("* cur  = Current count                                                  *");
        System.out.println("* min  = Lower limit                                                    *");
        System.out.println("* max  = Upper limit                                                    *");
        System.out.println("* avl  = Total number of available threads                              *");
        System.out.println("* busy = Number of busy threads                                         *");
        System.out.println("* add  = total number of work items added                               *");
        System.out.println("*************************************************************************");
    }


}
    
    

