/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2007 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.cli.framework;

import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;
import java.util.Set;


/**
 *    OptionMap Class
 *    @version  $Revision: 1.1 $
 */
public final class OptionsMap
{

    private Map<String, Value> optionsMap = null;

    public OptionsMap()
    {
        optionsMap = new HashMap<String, Value>();
    }
    

    
        /**
         * return option value
         */
    public String getOption(final String name)
    {
        if (!optionsMap.containsKey(name)) return null;
        return optionsMap.get(name).getValue();
    }


        /**
         *  return all options
         */
    public Map<String, String> getOptions()
    {
        Map<String, String> tempOptions = new HashMap<String, String>();
        final Iterator<String> optionNames = optionsMap.keySet().iterator();
        while (optionNames.hasNext())
        {
            final String optionKey = (String) optionNames.next();
            tempOptions.put(optionKey, optionsMap.get(optionKey).getValue());
        }
        return tempOptions;
    }
    

        /**
         * return all the command line options
         **/
    public Map<String,String> getCLOptions()
    {
        return getOptions(OptionValueFrom.CommandLine);
    }

    
        /**
         * return all the environment options
         **/
    public Map<String, String> getEnvOptions()
    {
        return getOptions(OptionValueFrom.Environment);
    }

    
        /**
         * return all the options with the default values
         **/
    public Map<String, String> getDefaultOptions()
    {
        return getOptions(OptionValueFrom.Default);
    }

    
        /**
         * return all the options that are set by thecommand module
         **/
    public Map<String, String> getOtherOptions()
    {
        return getOptions(OptionValueFrom.Other);
    }

    
    private Map<String, String> getOptions(final OptionValueFrom ovf)
    {
        Map<String, String> tempOptions = new HashMap<String, String>();
        final Iterator<String> optionNames = optionsMap.keySet().iterator();
        while (optionNames.hasNext())
        {
            final String optionKey = (String) optionNames.next();
            if (optionsMap.get(optionKey).getOptionValueFrom() == ovf)
            {
                tempOptions.put(optionKey, optionsMap.get(optionKey).getValue());
            }
        }
        return tempOptions;
    }
    

    /**
       add option to optionsMap.  If option name already exists, it
       will be replaced.
       @param name  option name
       @param value   option value
       @param ovf  option value from 
     */
    public void addOptionValue(final String name, final String value,
                          final OptionValueFrom ovf) 
    {
        optionsMap.put(name, new Value(value, ovf));
    }

    
    /**
       add option to optionsMap.  If option name already exists, it
       will be replaced.
       @param name  option name
       @param value   option value
     */
    public void addOptionValue(final String name, final String value)
    {
        optionsMap.put(name, new Value(value, OptionValueFrom.Other));
    }


    
    /**
       add CL option to optionsMap.  If option name already exists, it
       will be replaced.
       @param name  option name
       @param value   option value
       *param ovf  option value from 
     */
    public void addCLValue(final String name, final String value)
    {
        optionsMap.put(name, new Value(value, OptionValueFrom.CommandLine));
    }


    /**
       add Environment option to optionsMap.  If option name already exists, it
       will be replaced.
       @param name  option name
       @param value   option value
       *param ovf  option value from 
     */
    public void addEnvValue(final String name, final String value)
    {
        optionsMap.put(name, new Value(value, OptionValueFrom.Environment));
    }

    
    /**
       add Default option to optionsMap.  If option name already exists, it
       will be replaced.
       @param name  option name
       @param value   option value
       *param ovf  option value from 
     */
    public void addDefaultValue(final String name, final String value)
    {
        optionsMap.put(name, new Value(value, OptionValueFrom.Default));
    }


    public void addPrefValue(final String name, final String value)
    {
        optionsMap.put(name, new Value(value, OptionValueFrom.PrefFile));
    }


    public boolean containsName(final String name)
    {
        return optionsMap.containsKey(name);
    }


    public Set<String> nameSet()
    {
        return optionsMap.keySet();
    }
    

        /**
         * remove the entry from optionsMay
         */
    public void remove(final String name)
    {
        optionsMap.remove(name);
    }

    
    public String toString()
    {
        final Iterator optionNames = optionsMap.keySet().iterator();
        StringBuffer strbuf = new StringBuffer();
        
        while (optionNames.hasNext())
        {
            final String optionKey = (String) optionNames.next();
            strbuf.append("<"+optionKey+","+optionsMap.get(optionKey).getValue()+">");
            strbuf.append("\n");
        }
        return strbuf.toString();
    }
    
    
    private static class Value 
    {
        private String value;
        private OptionValueFrom ovf;

        Value(final String value, final OptionValueFrom ovf)
        {
            this.value = value;
            this.ovf = ovf;
        }
        

        /**
           Gets the value of the option
           @return the value of the option
        */
        String getValue()
        {
            return value;
        }
    
        /**
           Gets the OptionValueFrom
           @return OptionValueFrom
        */
        public OptionValueFrom getOptionValueFrom()
        {
            return ovf;
        }
    }

        //values from "Other" is for option valuel set during runtime from the Command module
    public enum OptionValueFrom { CommandLine, Environment, Default, PrefFile, Other}
}
