/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * AbstractTableAccessObject.java
 *
 * Created on July 13, 2005, 6:19 PM
 */

package com.sun.enterprise.admin.monitor.callflow;

import java.sql.BatchUpdateException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import com.sun.enterprise.server.ApplicationServer;
import com.sun.enterprise.server.ServerContext;
import java.util.logging.Level;
import java.util.logging.Logger;
import com.sun.enterprise.admin.common.constant.AdminConstants;
/**
 *
 * @author Harpreet Singh
 */
public abstract class AbstractTableAccessObject implements TableAccessObject{
    
    private static final Logger logger =
 		Logger.getLogger(AdminConstants.kLoggerName); 
    private static final String SYSTEM_PROPERTY =
            "com.sun.enterprise.callflow.trace";
    
    public boolean traceOn =
            System.getProperty(SYSTEM_PROPERTY, "false").equals("true");
    /*
     * SQL 99 error code for a table that already exists
     */
    private static final String TABLE_EXISTS_SQL_ERROR_CODE = "X0Y32";
    
    /*
     * An equivalent XOPEN SQL State for table already exists should be present
     * but could not be found.
     */
    
    Connection con = null;
    /* Holds the table name specific to this server instance. 
     * For e.g. REQUEST_START_TBL on a server instance with name "foo" will be
     * REQUEST_START_TBL__FOO. 
     */
    String tableName = null;
    private static final String DEFAULT_SERVER_NAME = "server";
    private long totalEntriesProcessed = 0;
    
    private String name = "AbstractTableAccessObject";
    
    abstract public boolean createTable(Connection connection);
    abstract public boolean dropTable(Connection connection);
   
   boolean createStatmentAndExecuteUpdate(String oldsql, 
           String tableNameWithoutServerInstance){
       
        String sql = updateSqlWithTableName (oldsql, tableNameWithoutServerInstance);
        boolean result = false;
        Statement stmt = null;
        try{
            if (con != null){
                stmt = con.createStatement();
                stmt.executeUpdate(sql);
                result = true;
            }
        } catch (java.sql.SQLException se) {
            // log it
            logger.log(Level.WARNING, "Error accessing CallFlow tables!", se);
            result = false;
        } finally {
            if(stmt != null){
                try{
                    stmt.close();
                }catch(java.sql.SQLException s){
                    // log it
                }
            }
            stmt = null;
        }
        return result;
   } 
  
   /**
    * This method is used to create a database table. If the table already 
    * exists, it logs a message and returns successfully.
    * As there is no mechanism to actually test if the database exists, it creates
    * the table and if there is an exception, it assumes it is due to table 
    * being present.
    */
   boolean createTable(String oldsql, 
           String tableNameWithoutServerInstance){
       
        String sql = updateSqlWithTableName (oldsql, tableNameWithoutServerInstance);
        boolean result = false;
        Statement stmt = null;
        try{
            if (con != null){
                stmt = con.createStatement();
                stmt.executeUpdate(sql);
                result = true;
            }
        } catch (java.sql.SQLException se) {
            // log it
            if (se.getSQLState().equalsIgnoreCase (TABLE_EXISTS_SQL_ERROR_CODE)){
                logger.log (Level.FINE, "callflow.table_already_exists_error", 
                        tableNameWithoutServerInstance);               
            } else {
                logger.log(Level.WARNING, "callflow.table_creation_error", 
                        tableNameWithoutServerInstance); 
                logger.log(Level.WARNING, "callflow.table_creation_error", se);
            }
            result = true;
        } finally {
            if(stmt != null){
                try{
                    stmt.close();
                }catch(java.sql.SQLException s){
                    // log it
                }
            }
            stmt = null;
        }
        return result;
   } 
   public String getServerInstanceName () {
       // get the server name from config
       String server = DEFAULT_SERVER_NAME;
       ServerContext sc = ApplicationServer.getServerContext();
        if (sc != null) {
           server =  sc.getInstanceName ();
        } 
        return "__" + server;
   } 
 
   /**
    * Adds the server instance name to the table names in the SQL statements 
    * for create/delete and insert. All creates and deletes need to call them 
    * before they submit the query to be executed
    * @param String complete sql that has the table name without the server instance
    * name
    * @param String Name of the table, this table name will be appended by 
    * a "__" and server name
    */
   String updateSqlWithTableName (String oldsql, String table) {
       String newsql = new String(oldsql);
       newsql = newsql.replaceAll(table, tableName);
       
        return newsql;       
   }
 
   public boolean delete(PreparedStatement pstmt, String[] requestId) {
        if (pstmt == null)
            return false;
        
        boolean result = false;
        try{
            for (int i = 0 ; i<requestId.length; i++) {
                pstmt.setString(1, requestId[i]);
                pstmt.addBatch();
            }
            int[] updated = pstmt.executeBatch();
            result =  (updated.length == requestId.length)? true : false;
            if (result == false){
                logger.log (Level.WARNING, "callflow.error_delete_row");                
            }
        }  catch(BatchUpdateException bue) {
            // log it
            logger.log (Level.WARNING, "callflow.error_delete_row");
            logger.log(Level.FINE, "Error data into CallFlow tables", bue);
            result = false;
        }catch (SQLException se) {
            // log it
            logger.log (Level.WARNING, "callflow.error_delete_row");
            logger.log(Level.FINE, "Error inserting data into CallFlow tables", se);
            result = false;
        }
        return result;        
        
    }      

    public boolean isTraceOn() {
        return traceOn;
    }

    public long getTotalEntriesProcessed() {
        return totalEntriesProcessed;
    }

    public void addTotalEntriesProcessed(int totalEntriesProcessed) {
        this.totalEntriesProcessed += totalEntriesProcessed;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }
}
