/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.jbi.serviceengine.core;
import com.sun.enterprise.jbi.serviceengine.ServiceEngineException;
import com.sun.enterprise.webservice.ServiceEngineUtil;
import com.sun.logging.LogDomains;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.management.ObjectName;
import javax.jbi.JBIException;
import javax.jbi.component.ComponentContext;
import javax.jbi.component.ComponentLifeCycle;


/**
 * Provide initialization, start, stop, and  shutdown processing. The JBI
 * implementation queries the component for the implementation of this
 * interface using the <code>Component.getComponentLifeCycle()</code> method.
 * The methods in this interface comprise the life cycle contract between the
 * JBI implementation and the component. The life cycle of a component begins
 * with a call to the <code>init()</code> method on an instance of the
 * component's implementation of this interface, and ends with the first call
 * to the <code>shutDown()</code> method on that instance. Between these two
 * calls, there can be any number of <code>stop()</code> and
 * <code>start()</code> calls.
 * @author Manisha Umbarje
 */
public class JavaEEServiceEngineLifeCycle implements ComponentLifeCycle {
    
    /**
     * This context provides access to data needed by all JBI components running in
     * the JBI environment.
     */
    private JavaEEServiceEngineContext context;
    private ComponentContext jbiContext;
    private Thread managerThread;
    
    /**
     * Internal handle to the logger instance
     */
    protected static Logger logger =
            LogDomains.getLogger(LogDomains.SERVER_LOGGER);
    
    /** Creates a new instance of JavaEEServiceEngineLifeCycle */
    public JavaEEServiceEngineLifeCycle() {
    }
    
    /**
     * Get the JMX <code>ObjectName</code> for any additional MBean for this
     * component. If there is none, return <code>null</code>.
     * @return the JMX object name of the additional MBean or <code>null</code>
     * if there is no additional MBean.
     */
    public ObjectName getExtensionMBeanName(){
        return null;
    }
    
    /**
     * Initialize the component. This performs initialization required by the
     * component but does not make it ready to process messages. This method
     * is called once for each life cycle of the component.
     * @param context the component's context.
     * @throws javax.jbi.JBIException if the component is unable to initialize.
     */
    public void init(ComponentContext jbiContext) throws JBIException {
        this.jbiContext = jbiContext;
    }
    
    /**
     * Shut down the component. This performs cleanup before the component is
     * terminated. Once this method has been called, <code>init()</code> must
     * be called before the component can be started again with a call to
     * <code>start()</code>.
     * @throws javax.jbi.JBIException if the component is unable to shut down.
     */
    public void shutDown() throws JBIException {
        
    }
    
    /**
     * Start the component. This makes the component ready to process messages.
     * This method is called after <code>init()</code> completes when the JBI
     * implementation is starting up, and when the component is being restarted
     * after a previous call to <code>shutDown()</code>. If <code>stop()</code>
     * was called previously but <code>shutDown()</code> was not, then
     * <code>start()</code> can be called again without another call to
     * <code>init()</code>.
     * @throws javax.jbi.JBIException if the component is unable to start.
     */
    public void start() throws JBIException{
        try {
            if(ServiceEngineUtil.isServiceEngineEnabled()) {
                logger.log(Level.FINE, "Service Engine starting");
                context = JavaEEServiceEngineContext.getInstance();
                context.setJBIContext(jbiContext);
                context.initialize();
                
                if(context.isSunESB()) {
                    jbiContext.getMBeanServer().registerMBean(
                            new JavaEEDeployer(),
                            jbiContext.getMBeanNames().createCustomComponentMBeanName("JavaEEDeployer"));
                    logger.log(Level.FINE, "Successfully registered JavaEEDeployer.");
                }

                // Starts all the threads such as thread accepting a message from
                // DeliveryChannel and delivering it to WorkManager
                managerThread = new Thread(context.getWorkManager());
                managerThread.start();
                logger.log(Level.INFO, "serviceengine.success_start");
                
            } else {
                logger.log(Level.INFO,
                        "Java EE Service Engine is not active as it is disabled " +
                        "by setting the JVM flag com.sun.enterprise.jbi.se.disable to true");
            }
            
            
        }catch(Exception e) {
            logger.log(Level.SEVERE, "serviceengine.error_start",
                    new Object[]{e.getMessage()});
                    throw new JBIException(e.getMessage());
        }
    }
    
    /**
     * Stop the component. This makes the component stop accepting messages for
     * processing. After a call to this method, <code>start()</code> can be
     * called again without first calling <code>init()</code>.
     * @throws javax.jbi.JBIException if the component is unable to stop.
     */
    public void stop() throws JBIException {
        //Stop multiple threads involved in request processing by the
        //JavaEEServiceEngine
        try {
            if(ServiceEngineUtil.isServiceEngineEnabled()) {
                context.getDeliveryChannel().close();
                context.getWorkManager().stop();
                managerThread.join();
                if(context.isSunESB()) {
                    jbiContext.getMBeanServer().unregisterMBean(
                        jbiContext.getMBeanNames().createCustomComponentMBeanName("JavaEEDeployer"));
                }
                logger.log(Level.INFO, "serviceengine.success_stop");
            }
        } catch(Exception e) {
            e.printStackTrace();
            throw new JBIException(e.getMessage());
        }
    }
    
}
