/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 
/*
 * $Header: /cvs/glassfish/appserv-api/src/java/com/sun/appserv/management/helper/RefHelper.java,v 1.2 2007/01/19 20:43:50 llc Exp $
 * $Revision: 1.2 $
 * $Date: 2007/01/19 20:43:50 $
 */
package com.sun.appserv.management.helper;

import java.util.Set;
import java.util.HashSet;
import java.util.Collections;

import javax.management.ObjectName;

import com.sun.appserv.management.base.AMX;
import com.sun.appserv.management.base.AMXDebug;
import com.sun.appserv.management.base.XTypes;
import com.sun.appserv.management.base.QueryMgr;
import com.sun.appserv.management.base.Util;

import com.sun.appserv.management.util.misc.CollectionUtil;
import com.sun.appserv.management.util.misc.GSetUtil;
import com.sun.appserv.management.util.misc.Output;

import com.sun.appserv.management.config.RefConfigReferent;
import com.sun.appserv.management.config.DeployedItemRefConfigReferent;
import com.sun.appserv.management.config.ResourceRefConfigReferent;
import com.sun.appserv.management.config.ServerRefConfigReferent;
import com.sun.appserv.management.config.ClusterRefConfigReferent;
import com.sun.appserv.management.config.ClusterRefConfig;
import com.sun.appserv.management.config.ServerRefConfig;
import com.sun.appserv.management.config.ConfigRemover;

import com.sun.appserv.management.config.RefConfig;
import com.sun.appserv.management.config.ServerRefConfig;
import com.sun.appserv.management.config.ClusterRefConfig;
import com.sun.appserv.management.config.ResourceConfig;
import com.sun.appserv.management.config.ResourceRefConfig;
import com.sun.appserv.management.config.ResourceRefConfigCR;
import com.sun.appserv.management.config.ModuleConfig;
import com.sun.appserv.management.config.DeployedItemRefConfig;
import com.sun.appserv.management.config.DeployedItemRefConfigCR;
import com.sun.appserv.management.config.LifecycleModuleConfig;
import com.sun.appserv.management.config.CustomMBeanConfig;

/**
    Helper routines for references.  This class should only be used
    in EE builds.
    
	@since AppServer 9.0
 */
public class RefHelper
{
    private	RefHelper()	{}


        private static Output
    getDebug()
    {
        return AMXDebug.getInstance().getOutput( "com.sun.appserv.management.helper.RefHelper" );
    }
    
    /**
        Find all ref configs of any j2EEType
        @return Set of all {@link RefConfig} found.
     */
        public static <T extends RefConfig> Set<T>
    findAllRefConfigs( final QueryMgr  queryMgr )
    {
        final Set<T>  refs    = 
            queryMgr.queryInterfaceSet( RefConfig.class.getName(), null);
        
        return refs;
    }
    
    /**
        Find all ref configs of the specified j2EEType
        @return Set of all {@link RefConfig} found.
     */
        public static <T extends RefConfig> Set<T>
    findAllRefConfigsByJ2EEType(
        final QueryMgr  queryMgr,
        final String    refJ2EEType )
    {
        final Set<T>  refs    =
            queryMgr.queryJ2EETypeSet( refJ2EEType );
        
        return refs;
    }
    
    /**
        Find all {@link ResourceRefConfig}.
        @return Set of all {@link ResourceRefConfig} found.
     */
        public static Set<ResourceRefConfig>
    findAllResourceRefConfigs( final QueryMgr  queryMgr )
    {
        return
            findAllRefConfigsByJ2EEType( queryMgr, XTypes.RESOURCE_REF_CONFIG );
    }
    
    /**
        Find all {@link DeployedItemRefConfig}.
        @return Set of all {@link DeployedItemRefConfig} found.
     */
        public static Set<DeployedItemRefConfig>
    findAllDeployedItemRefConfigs( final QueryMgr  queryMgr )
    {
        return
            findAllRefConfigsByJ2EEType( queryMgr, XTypes.DEPLOYED_ITEM_REF_CONFIG );
    }
    
    /**
        Find all {@link ServerRefConfig}.
        @return Set of all {@link ServerRefConfig} found.
     */
        public static Set<ServerRefConfig>
    findAllServerRefConfigs( final QueryMgr  queryMgr )
    {
        return
            findAllRefConfigsByJ2EEType( queryMgr, XTypes.SERVER_REF_CONFIG );
    }
    
    
    /**
        Find all {@link ClusterRefConfig}.
        @return Set of all {@link ClusterRefConfig} found.
     */
        public static Set<ClusterRefConfig>
    findAllClusterRefConfigs( final QueryMgr  queryMgr )
    {
        return
            findAllRefConfigsByJ2EEType( queryMgr, XTypes.CLUSTER_REF_CONFIG );
    }
    
    /**
        Find all {@link RefConfig} of the specified j2EEType having
        the specified name.
        @return Set of all {@link RefConfig} found with the specified name.
     */
        public static <T extends RefConfig> Set<T>
    findAllRefConfigsWithName(
        final QueryMgr  queryMgr,
        final String    refJ2EEType,
        final String    name )
    {
        final String props = Util.makeRequiredProps( refJ2EEType, name );
        
        final Set<T> refs  = queryMgr.queryPatternSet( null, props );
        
        return refs;
    }
    
    /**
        @return the j2eeType of the config element which can <i>refer</i> to this item
     */
        public static String
    getReferentRefJ2EEType( final RefConfigReferent referent )
    {
        String  j2eeType    = null;
        
        if ( referent instanceof ResourceRefConfigReferent )
        {
            j2eeType    = XTypes.RESOURCE_REF_CONFIG;
        }
        else if ( referent instanceof DeployedItemRefConfigReferent )
        {
            j2eeType    = XTypes.DEPLOYED_ITEM_REF_CONFIG;
        }
        else if ( referent instanceof ServerRefConfigReferent )
        {
            j2eeType    = XTypes.SERVER_REF_CONFIG;
        }
        else if ( referent instanceof ClusterRefConfigReferent )
        {
            j2eeType    = XTypes.CLUSTER_REF_CONFIG;
        }
        else
        {
            throw new IllegalArgumentException(
                "Unknown referent class: " + Util.asAMX( referent ).getJ2EEType() );
        }
        
        return j2eeType;
    }
    
   private static final Set<String>  REFERENT_J2EE_TYPES =
    Collections.unmodifiableSet( GSetUtil.newSet( new String[]
        {
            XTypes.RESOURCE_REF_CONFIG,
            XTypes.DEPLOYED_ITEM_REF_CONFIG,
            XTypes.SERVER_REF_CONFIG,
            XTypes.CLUSTER_REF_CONFIG,
        }));
    
        public static Set<String>
    getReferentJ2EETypes()
    {
        return REFERENT_J2EE_TYPES;
    }
   
    
     /**
        Find all {@link RefConfig} that reference the specified j2eeType/name
        combo.
     */
        public static <T extends RefConfig> Set<T>
    findAllRefConfigs( final RefConfigReferent    referent )
    {
        final AMX   amx = Util.asAMX( referent );
        
        return findAllRefConfigsWithName( getQueryMgr( amx ),
                    getReferentRefJ2EEType( referent ),
                    amx.getName() );
    }
    
    /**
        Find all {@link RefConfig} that reference the specified j2eeType/name
        combo.
     */
        public static <T extends RefConfig> Set<T>
    findAllRefConfigs( final RefConfigReferent referent, final String refJ2EEType )
    {
        final AMX   amx = Util.asAMX( referent );
        
        return findAllRefConfigsWithName( getQueryMgr( amx ), refJ2EEType, amx.getName() );
    }
    
    /**
        Find all {@link DeployedItemRefConfig} that reference
        the specified item.
     */
        public static Set<DeployedItemRefConfig>
    findAllRefConfigs( final DeployedItemRefConfigReferent referent )
    {
        final AMX   amx = Util.asAMX( referent );
        
        return
            findAllRefConfigsWithName( getQueryMgr( amx ), XTypes.DEPLOYED_ITEM_REF_CONFIG, amx.getName() );
    }
    
    /**
        Find all {@link ResourceRefConfig} that reference
        the specified item.
     */
        public static Set<ResourceRefConfig>
    findAllRefConfigs( final ResourceRefConfigReferent referent )
    {
        final AMX   amx = Util.asAMX( referent );
        
        return
            findAllRefConfigsWithName( getQueryMgr( amx ), XTypes.RESOURCE_REF_CONFIG, amx.getName() );
    }
    
    /**
        Find all {@link ServerRefConfig} that reference
        the specified item.
     */
        public static Set<ServerRefConfig>
    findAllRefConfigs( final ServerRefConfigReferent referent )
    {
        final AMX   amx = Util.asAMX( referent );
        
        return
            findAllRefConfigsWithName( getQueryMgr( amx ), XTypes.SERVER_REF_CONFIG, amx.getName() );
    }
    
        private static QueryMgr
    getQueryMgr( final AMX amx )
    {
        return amx.getDomainRoot().getQueryMgr();
    }
    
    /**
        Find all {@link ClusterRefConfig} that reference
        the specified item.
     */
        public static Set<ClusterRefConfig>
    findAllRefConfigs( final ClusterRefConfigReferent referent )
    {
        final AMX   amx = Util.asAMX( referent );
        
        return
            findAllRefConfigsWithName( getQueryMgr( amx ), XTypes.CLUSTER_REF_CONFIG, amx.getName() );
    }
    
        public static <T extends RefConfig> Set<T>
    removeAllRefsTo(
        final RefConfigReferent item,
        final boolean allowSingleFailure )
    {
        final Set<T> refs   = RefHelper.findAllRefConfigs( item );

	    final Set<T>    failures   = RefHelper.removeRefConfigs( refs );
	    if ( failures.size() != 0 )
        {
            if ( failures.size() > 1 || ! allowSingleFailure )
            {
                final Set<ObjectName>  objectNames = Util.toObjectNames( failures );
                
                throw new IllegalArgumentException( "failure removing refererences:\n{" +
                    CollectionUtil.toString( objectNames, "\n" ) + "\n}" );
            }
        }
        
        return failures;
    }
    
    /**
        Remove all specified references (that are possible).
        @return any references that could not be removed
     */
        public static <T extends RefConfig> Set<T>
    removeRefConfigs( final Set<T> refs )
    {
        final Set<T> failures    = new HashSet<T>();
        
        for( final T ref : refs )
        {
            final ConfigRemover rrc   = ConfigRemover.class.cast( ref.getContainer() );
            
            try
            {
                // may fail as illegal in PE or EE with one ref
                rrc.removeConfig( ref.getJ2EEType(), ref.getName() );
            }
            catch( Exception e )
            {
                failures.add( ref );
            }
        }
        
        return failures;
    }
    
}












