/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.appserv.management.util.misc;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ExecutorService;

import com.sun.appserv.management.helper.AMXDebugHelper;
import com.sun.appserv.management.util.misc.StringUtil;
import com.sun.appserv.management.util.misc.Timings;


/**
    <b>INTERNAL USE ONLY -- DO NOT USE</b>
    Base class (can be used directly) for running small, short lived tasks.  An ExecutorService
    is used for efficiency, and excess threads are discarded quickly.
    <p>
    Includes convenience routines for submitting tasks, determining result status.
    <p>
    <b>Example (inline usage)</b>
    <pre>
    final RunnableBase myTask = new RunnableBase( "Compute PI" ) {
        public void run() {
            final double pi = 3.141926386; // cheater method
        }
    };
    myTask.submit();
    ...
    // wait for compute task
    myTask.waitDoneThrow(); // or waitDone() followed by myTask.getThrowable()
    </pre>
    <p>
    <b>NOTE: </b>An ExecutorService is used with a thread pool.  Inheritable thread
    local variables will <b>not</b> be inherited.
 */
public abstract class  RunnableBase<T> implements Runnable
{
    /** a Throwable if anything was thrown from the run loop */
    private volatile Throwable      mThrowable;
    
    /** means to block client threads until done */
    private CountDownLatch          mLatch;
    
    /** optional name of the task */
    private final String            mName;
    
    /** debugging: whether to sleep a random amount.  See {@link #setUseRandomSleep} */
    private volatile boolean        mUseRandomSleep;
    
    // optional data for use by the task
    private final T                 mData;
    
    private volatile long           mSubmitNanos;
    private volatile long           mRunStartNanos;
    private volatile long           mRunDoneNanos;
    
    private final AMXDebugHelper    mDebug;
    
    /** execute the task synchronously (in calling thread) */
    public static final int    SUBMIT_SYNC = 0;
    
    /** execute the task asynchronously (separate thread), but wait till done */
    public static final int    SUBMIT_ASYNC_WAIT = 1;
    
    /** execute the task asynchronously (separate thread), return immediately */
    public static final int    SUBMIT_ASYNC = 2;
    
        private void
    debug( final Object... args)
    {
        if ( mDebug.getDebug() )
        {
            mDebug.println( args );
        }
    }
    
    private static final ExecutorService   _Exec = Executors.newCachedThreadPool();
        private void
    _submit( final RunnableBase r, final int howToRun )
    {
        if ( mLatch != null )
        {
            // already in progress
            throw new IllegalStateException();
        }
        if ( howToRun != SUBMIT_SYNC && howToRun != SUBMIT_ASYNC && howToRun != SUBMIT_ASYNC_WAIT )
        {
            throw new IllegalArgumentException();
        }
        
        r.mSubmitNanos    = System.nanoTime();
        
        if ( howToRun == SUBMIT_SYNC )
        {
            run();
        }
        else
        {
            //final long start = System.nanoTime();
            mLatch  = new CountDownLatch(1);
            //debug( "TIME TO CREATE LATCH: " + (System.nanoTime() - start ) );
            _Exec.submit( r );
            
            if ( howToRun == SUBMIT_ASYNC_WAIT )
            {
                //debug( "RunnableBase.submit(): waitTillDone() for " + StringUtil.quote( getName() ));
                //final long start = System.nanoTime();
                waitDoneThrow();
                //debug( "RunnableBase.submit(): waitTillDone() completed for " +
                   // StringUtil.quote( getName() ) + " in " + StringUtil.getTimingString(System.nanoTime() - start) );
            }
        }
    }
    
    /**
        Submit the task for execution.  The task might finish before this method returns or
        afterwards. See {@link #waitDone} and {@link #waitDoneThrow}.
     */
        public void
    submit( )
    {
        _submit( this, SUBMIT_ASYNC );
    }
    
       
    /**
        Submit the task for execution with {@link #submit()}.  If 'waitTillDone'
        is true, then this method won't return until the task has finished.  This
        method is useful as a transition method in the course of converting from
        serialized execution to threaded execution, allowing a simple boolean switch
        to make the change in behavior.<p>
        The task is still executed in its own thread, so as to produce the same
        runtime environment that would be used for asynchronous execution (eg thread-local
        variables).
        @param waitTillDone if true, the method executes synchronously
     */
        public void
    submit( final int howToRun )
    {
        _submit( this, howToRun );
    }
        /**
        Create a new task.
        @param name use-readable name of the task
        @param data optional arbitrary data (see {@link #getData})
     */
        protected
    RunnableBase( final String name, final T data )
    {
        mDebug  = new AMXDebugHelper( "RunnableBase-" + name );
        mDebug.setEchoToStdOut( true );
        
        mName       = name == null ? (this.getClass().getName() + ".<no_name>") : name ;
        mData       = data;
        mThrowable  = null;
        mUseRandomSleep = true;
        mSubmitNanos    = 0;
        mRunStartNanos    = 0;
        mRunDoneNanos    = 0;
        
        mLatch   = null;
    }
    
        protected
    RunnableBase( final String name )
    {
        this( name, null );
    }
    
    public final T    getData()   { return mData; }
    
        protected
    RunnableBase(  )
    {
        this( null );
    }
    
    /* subclass must implement doRun() */
    protected abstract void doRun() throws Exception;
    
        public String
    getName()
    {
        return (mName == null || mName.length() == 0) ? this.getClass().getName() : mName;
    }
    
        protected  static void
    sleepMillis( final long millis )
    {
        try
        {
            Thread.sleep( millis );
        }
        catch( InterruptedException e )
        {
        }
    } 
    
    private static final long MAX_RANDOM_SLEEP_MILLIS   = 500;
    
    /**
        Good for debugging timing issues; a task will insert an artificial delay
        by a random amount.
     */
        public void 
    setUseRandomSleep( final boolean useRandom )
    {
        mUseRandomSleep = useRandom;
    }
    
    // this way, we don't have to execute a getTimings() call, which is synchronized
    private static final Timings TIMINGS    = Timings.getInstance( "RunnableBase" );
        public static Timings
    getTimings()
    {
        return TIMINGS;
    }
    
    /**
        May be called synchronously or via another thread {@link #submit}.
        See {@link #waitDone} and {@link #waitDoneThrow} and {@link #getThrowable}.
     */
        public final void
    run()
    {
        mRunStartNanos    = System.nanoTime();
        
        if ( mUseRandomSleep )
        {
            sleepMillis( mRunStartNanos % MAX_RANDOM_SLEEP_MILLIS );
        }
        try
        {
            doRun();
        }
        catch( Throwable t )
        {
            mThrowable  = t;
        }
        finally
        {
            mRunDoneNanos    = System.nanoTime();
            //debug( toString() );
            if ( mLatch != null )
            {
                mLatch.countDown();
            }
            // do this after we release the latch
            final String msg = "RunnableBase-" + StringUtil.quote(getName());
            final long runTime    = getNanosFromSubmit();
      // final long start = System.nanoTime();
            getTimings().add( msg, runTime);
       //debug( "TIME TO ADD TIMING: " + (System.nanoTime() - start ) );
        }
        mLatch  = null; // no longer usable, get rid of our ref
    }
    
    /**
        @return the number of nanoseconds to execute the task from the time it was submitted
     */
        public long
    getNanosFromSubmit()
    {
        return mRunDoneNanos - mSubmitNanos;
    }
    
    /**
        @return the number of nanoseconds to execute the task from the time it actually entered
        the {@link #run} method.
     */
        public long
    getNanosFromRunStart()
    {
        return mRunDoneNanos - mRunStartNanos;
    }
    
    /**
        @return the number of nanoseconds between task-submittal and the actual execution start
     */
        public long
    getRunLatency()
    {
        return mRunStartNanos - mSubmitNanos;
    }
    
    /**
        Block until the task has finished, and return any Throwable (hopefully null).
        @return the Throwable that was thrown (if any), otherwise null
     */
        public final Throwable
    waitDone()
    {
        // if mLatch is null, it was run synchronously, or has already finished (or never started)
        // use temp, avoid race condition between null check and usage
        final CountDownLatch latch  = mLatch;
        if ( latch != null )
        {
            try
            {
                latch.await();
            }
            catch( final InterruptedException intr )
            {
               throw new RuntimeException( intr );
            }
        }
        return mThrowable;
    }
    
    /**
        Block until the task has finished.  If a Throwable was thrown, then this method
        will rethrow it, or a RuntimeException.
     */
        public final void
    waitDoneThrow()
    {
        final Throwable t   = waitDone();
        if ( t != null )
        {
            if ( t instanceof RuntimeException )
            {
                throw (RuntimeException)t;
            }
            else if ( t instanceof Error )
            {
                throw (Error)t;
            }
            else
            {
                throw new RuntimeException( t );
            }
        }
    }
        
        public String
    toString()
    {
        final String delim = ", ";
        
        final boolean started   = mSubmitNanos != 0;
        final boolean done      = mRunDoneNanos != 0;
        final long runTimeNanos      = started ?
            (done ? (mRunDoneNanos - mRunStartNanos) : System.nanoTime() - mRunStartNanos) : 0;
        final String throwable = mThrowable == null ? "" : mThrowable.toString();
        
        final String runTimeString = StringUtil.getTimingString( runTimeNanos );
        
        return "Runnable \"" + this.getClass().getName() + "\"" + delim + "name = " + getName() +
            delim + "started=" + started + delim + "done=" + done +
            delim + "run-time=" + runTimeString + delim + throwable;
    }
};
































