/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.appserv.management.util.misc;

import java.util.Set;
import java.util.Map;
import javax.management.ObjectName;
import javax.management.Attribute;
import javax.management.AttributeList;

import com.sun.appserv.management.util.stringifier.SmartStringifier;
import com.sun.appserv.management.util.jmx.JMXUtil;


/**
	Escapes/unescapes strings
 */
public final class StringUtil
{
	private	StringUtil()	{}
	
	public final static char	QUOTE_CHAR	= '\"';
	public final static String	QUOTE		= "" + QUOTE_CHAR;
	
	/**
		Line separator as returned by System.getProperty()
	 */
	public final static String	LS		= System.getProperty( "line.separator", "\n" );
	
		public static String
	quote( Object o )
	{
		return( quote( o, QUOTE_CHAR ) );
	}
	
		public static String
	quote( Object o,  char leftHandChar )
	{
		final String	s	= o == null ? "null" : SmartStringifier.toString( o );
		
		char	leftChar	= leftHandChar;
		char	rightChar	= leftHandChar;
		
		if ( leftHandChar == '(' )
		{
			rightChar	= ')';
		}
		else if ( leftHandChar == '{' )
		{
			rightChar	= '}';
		}
		else if ( leftHandChar == '[' )
		{
			rightChar	= ']';
		}
		else if ( leftHandChar == '<' )
		{
			rightChar	= '>';
		}
		else
		{
			// same char on both left and right
		}
		
		final String	out	= leftChar + s + rightChar;

		return( out );
	}
	
	
		public static String
	toHexString( byte theByte )
	{
		String result	= Integer.toHexString( ((int)theByte) & 0x000000FF );
		if ( result.length() == 1 )
		{
			result	= "0" + result;
		}
		return( result );
	}
	
		public static String
	toHexString( byte[] bytes )
	{
		return( toHexString( bytes, null ) );
	}
	
		public static String
	toHexString( byte[] bytes, String delim )
	{
		final StringBuffer	buf	= new StringBuffer();
		
		if ( bytes.length == 0 )
		{
			// nothing
		}
		else if ( delim == null || delim.length() == 0 )
		{
			for( int i = 0; i < bytes.length; ++i )
			{
				buf.append( toHexString( bytes[ i ] ) );
			}
		}
		else
		{
			for( int i = 0; i < bytes.length; ++i )
			{
				buf.append( toHexString( bytes[ i ] ) + delim );
			}
		
			// remove trailing delim
			buf.setLength( buf.length() - 1);
		}
		
		return( buf.toString() );
	}
	
		public static String
	stripSuffix(
		final String s,
		final String suffix )
	{
		String	result	= s;
		
		if ( s.endsWith( suffix ) )
		{
			result	= s.substring( 0, s.length() - suffix.length() );
		}
		
		return( result );
	}
	
	
		public static String
	replaceSuffix(
		final String s,
		final String fromSuffix,
		final String toSuffix )
	{
		if ( ! s.endsWith( fromSuffix ) )
		{
			throw new IllegalArgumentException( fromSuffix );
		}

		return( stripSuffix( s, fromSuffix ) + toSuffix );
	}
	
		public static String
	stripPrefix(
		final String s,
		final String prefix )
	{
		String	result	= s;
		
		if ( s.startsWith( prefix ) )
		{
			result	= s.substring( prefix.length(), s.length() );
		}
		
		return( result );
	}
	
	
		public static String
	stripPrefixAndSuffix(
		final String s,
		final String prefix,
		final String suffix )
	{
		return stripPrefix( stripSuffix( s, suffix ), prefix );
	}
	

		public static String
	upperCaseFirstLetter( final String s)
	{
		String	result	= s;
		
		if ( s.length() >= 1 )
		{
			result	= s.substring( 0, 1 ).toUpperCase() + s.substring( 1, s.length() );
		}

		return( result );
	}
	
	    
           public static String
    toString( final Object o )
    {
        String  s   = null;
        
        if ( o instanceof String )
        {
            s   = (String)o;
        }
        else if ( o instanceof Throwable )
        {
            s   = ExceptionUtil.toString( (Throwable)o );
        }
        else if ( o instanceof ObjectName )
        {
            s   = JMXUtil.toString( (ObjectName)o );
        }
        else if ( o instanceof Attribute )
        {
            final Attribute a = (Attribute)o;
            s   = a.getName() + "=" + toString( a.getValue() );
        }
        else if ( o instanceof AttributeList )
        {
            final Map<String,Object>    items   = JMXUtil.attributeListToValueMap( (AttributeList)o );
            s   = "{" + MapUtil.toString( items ) + "}";
        }
        else if ( o instanceof byte[] )
        {
            final byte[]    b   = byte[].class.cast( o );
            s  = "byte[] of length " + b.length;
        }
        else if ( o == null )
        {
            s   = "null";
        }
        else if ( o instanceof Object[] )
        {
            s   = toString( ", ", (Object[])o );
        }
        else
        {
            s   = "" + o;
        }
        
        return s;
    }

    
        public static String
    toString( final String[] args )
    {
        return toString( ", ", args );
    }
    
        public static String
    toString( final String delim, final String... args )
    {
        return toString( delim, (Object[])args );
    }
    
	/**
	    Turn an array (or varargs) set of Objects into a String
	    using the specified delimiter.
	 */
	    public static String
    toString( final String delim, final Object... args )
    {
        String  result  = null;
        
        if ( args == null )
        {
            result  = "" + null;
        }
        else if ( args.length == 0 )
        {
            result  = "";
        }
        else if ( args.length == 1 )
        {
            result  = toString( args[ 0 ] );
        }
        else
        {
            final StringBuilder builder = new StringBuilder();
            
            for( int i = 0; i < args.length - 1; ++i )
            {
                builder.append( toString( args[ i ] ) );
                builder.append( delim );
            }
            builder.append( toString( args[ args.length - 1 ] ) );
            
            result  = builder.toString();
        }
        
        
        return result;
     }
     
    /**
        @return the prefix found, or null if not found
     */
        public static String
    getPrefix(
        final Set<String>   prefixes,
        final String        s )
    {
        String  result  = null;
        for( final String prefix : prefixes )
        {
            if ( s.startsWith( prefix ) )
            {
                result  = prefix;
                break;
            }
        }
        return result;
    }
    
    /**
        @return the String after stripping the prefix
        @throws IllegalArgumentException if no prefix found
     */
        public static String
    findAndStripPrefix(
        final Set<String>   prefixes,
        final String        s )
    {
        final String  prefix    = getPrefix( prefixes, s );
        if ( prefix == null )
        {
            throw new IllegalArgumentException( s );
        }
        
        return stripPrefix( s, prefix );
    }
    
	private static String	NEWLINE_STR		= null;
	
	    public static String
	NEWLINE()
	{
	    if ( NEWLINE_STR == null )
	    {
	        NEWLINE_STR = System.getProperty( "line.separator" );
	    }
	    return NEWLINE_STR;
	}
   
   
        private static String
    zeroPad( final long num )
    {
        String  result  = null;
        if ( num >= 100 )
            result  = "" + num;
        else if ( num >= 10 )
            result  = "0" + num;
        else
            result  = "00" + num;
        return result;
    }
    
    /**
        @param nanos    elapsed nanoseconds
        @return a String describing the elapsed duration
     */
        public static String
    getSecondsString( final long nanos )
    {
        final long seconds     = nanos / (1000 * 1000 * 1000 );
        final long millis      = nanos % (1000 * 1000 * 1000 ) / ( 1000 * 1000);
        final long micros      = nanos % (1000 * 1000 ) / 1000;
        final long nanosMod    = nanos % 1000;
        
        String runTimeString = seconds + "." + zeroPad(millis)  + zeroPad(micros) ;
        
        if ( nanosMod != 0 )
        {
            runTimeString   = runTimeString + zeroPad(nanosMod);
        }
        return runTimeString + " seconds";
    }
    
       /**
        @param nanos    elapsed nanoseconds
        @return a String describing the elapsed duration
     */
        public static String
    getTimingString( final long nanos )
    {
        final long seconds     = nanos / (1000 * 1000 * 1000 );
        final long millis      = nanos % (1000 * 1000 * 1000 ) / ( 1000 * 1000);
        final long micros      = nanos % (1000 * 1000 ) / 1000;
        final long nanosMod    = nanos % 1000;
        
        String runTimeString = null;
        final long MICROSECOND = 1000;
        final long MILLISECOND = 1000 * MICROSECOND;
        if ( nanos < 10 * 1000 )
        {
            runTimeString   = nanos + " nanoseconds";
        }
        else if ( nanos < 1000 * 10 * 1000 )
        {
            runTimeString   = (nanos / MICROSECOND) + " microseconds";
        }
        else
        {
            runTimeString   = (nanos / MILLISECOND ) + " milliseconds";
        }
        
        return runTimeString + " seconds";
    }


}
















