/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment;

import java.util.*;
import java.io.*;
import java.io.Serializable; 
import com.sun.enterprise.util.NotificationListener;
import com.sun.enterprise.util.NotificationEvent;
import com.sun.enterprise.deployment.util.DescriptorVisitor;

    /**
    * Descriptor is the root class for all objects
    * representing deployment information in J2EE. Descriptors
    * notifiy listeners of state changes, and have a name, description,
    * and icons.
    * 
    *@author Danny Coward
    */

public class Descriptor extends DynamicAttributesDescriptor implements Serializable {

    /** Notification String for a general descriptor change. */
    public static final String DESCRIPTOR_CHANGED  = "Descriptor change";
    public static final String NAME_CHANGED        = "NameChanged";
    public static final String DESCRIPTION_CHANGED = "DescriptionChanged";
    public static final String LARGE_ICON_CHANGED  = "LargeIconChanged";
    public static final String SMALL_ICON_CHANGED  = "SmallIconChanged";

    /** static flag to indicate descriptors should bounds check. */
    private static boolean boundsChecking = true;
    
    /** My Vector of NotificationListener objects. */
    protected Vector listeners = new Vector();

    /** My display name indexed by language */
    private Map displayNames = null;
    
    /** My descriptions indexed by language */
    private Map descriptions = null;
    
    /** icons map indexed by language */
    private Map largeIcons = null;
    private Map smallIcons = null;
    
        
    /**
    * The default constructor. Constructs a descriptor with 
    * name, description and icons as empty Strings.
    *
    */
    public Descriptor() {
	this.listeners = new Vector();
    } 
    
    /**
    * The copy constructor.
    */
    protected Descriptor(Descriptor other) {
        if (other.displayNames!=null)
	this.displayNames = new HashMap(other.displayNames);
        if (other.descriptions!=null) 
            this.descriptions = new HashMap(other.descriptions);
        if (other.largeIcons!=null)
            this.largeIcons = new HashMap(other.largeIcons);
        if (other.smallIcons!=null)
            this.smallIcons = new HashMap(other.smallIcons);
    }
    
    /**
    * Constructs a descriptor with given
    * name, description.
    *
    *  @param name the name of the descriptor. 
    *  @param description the name of the descriptor.  
    *
    */
    public Descriptor(String name, String description) {
	this();
	setLocalizedDisplayName(null, name);
	setLocalizedDescription(null, description);
    }
    
    /**
    * Sets a global flag to enable or disable boudsn checking
    * of deployment information
    *
    *  @param b true for bounds checking on, false else.
    *
    */
    public static void setBoundsChecking(boolean b) {
	boundsChecking = b;
    }
    
    /**
    * Answers whether the object model is bounds checking.
    *
    *  @return true for boudsn checking, false else.
    *
    */
    public static boolean isBoundsChecking() {
	return boundsChecking;
    }
    
    /**
    * Sets the name of this descriptor.
    *
    *  @param name the new name of the descriptor. 
    *
    */
    public void setName(String name) {
	setLocalizedDisplayName(null, name);
    }
    
    /**
    * The name of this descriptor as a String.
    *
    *  @return the name of this descriptor 
    *
    */
    public String getName() {
	return getLocalizedDisplayName(null);
    }
    
    /**
     * Add a localized display name for this descriptor
     * @param the local identifier (null if using default locale)
     * @param the localized string
     */
    public void setLocalizedDisplayName(String lang, String displayName) {
        
        if (lang==null) {
            lang = Locale.getDefault().getLanguage();
        }
        if (displayNames==null) {
            displayNames = new HashMap();
        }
        displayNames.put(lang, displayName);
        changed(NAME_CHANGED);
    }
    
    /**
     * @return the localized display name for the passed language
     * @param the language 
     */
    public String getLocalizedDisplayName(String language) {
        
        if (displayNames==null) {
            return "";
        }
        
        String originalLanguage = language;        
        if (language==null) {
            language=Locale.getDefault().getLanguage();
        }
        
        String localizedName = (String) displayNames.get(language);
        if (localizedName!=null) {
            return localizedName;
        }

        // so far, no luck, it is possible that this
        // environment property was transfered through jndi 
        // between machines with different locales, if I have
        // at least one value, and no language was specified,
        // let's return it.
        if (originalLanguage==null && displayNames.size()>0) {
            return (String) displayNames.values().iterator().next();
        }
        // all other cases return empty strings
        return "";
    }
    
    /**
     * @return the localized display name indexed by language
     */
    public Map getLocalizedDisplayNames() {
        return displayNames;
    }
    
    /**
     * sets the display name for this bundle
     */
    public void setDisplayName(String name) {
        setName(name);
    }
    
    /** 
     * @return the display name
     */
    public String getDisplayName() {
        return getName();
    }    
    
    /**
    * Sets the description text of this descriptor.
    *
    *  @param description the new description text of the descriptor. 
    *
    */
    public void setDescription(String description) {
        setLocalizedDescription(null, description);
    }
    
    /**
    * The description text of this descriptor as a String.
    *
    *  @return the description text of this descriptor 
    *
    */
    public String getDescription() {
	return getLocalizedDescription(null);
    }
    
    /**
     * Add a localized description for this descriptor
     * @param the local identifier (null if using default locale)
     * @param the localized string
     */
    public void setLocalizedDescription(String lang, String description) {
        if (lang==null) {
            lang = Locale.getDefault().getLanguage();
        }
        if (descriptions==null) {
            descriptions = new HashMap();
        }
        descriptions.put(lang, description);
        changed(DESCRIPTION_CHANGED);
    }
    
    /**
     * @return the localized description
     * @param the local language
     */
    public String getLocalizedDescription(String lang) {
        
        if (descriptions==null)
            return "";       
        if (lang==null) {
            lang = Locale.getDefault().getLanguage();
        }
        String description = (String) descriptions.get(lang);
        if (description==null) {
            return "";
        }
        return description;
    }
    
    /** 
     * @return a Map of localized description, where lang is the key 
     */
    public Map getLocalizedDescriptions() {
        return descriptions;
    }
    
    /**
     * Sets the large icon uri for a particular language
     * @param the language identifier
     * @param the large icon uri
     */
    public void setLocalizedLargeIconUri(String lang, String uri) {
        if (lang==null) {
            lang = Locale.getDefault().getLanguage();
        }
        if (largeIcons==null) {
            largeIcons = new HashMap();
        }
        largeIcons.put(lang, uri);
        changed(LARGE_ICON_CHANGED);
    }
    
    /**
     * @return the large icon uri for a language
     * @param the language or null for the current locale
     */
    public String getLocalizedLargeIconUri(String lang) {
        if (largeIcons==null) {
            return null;
        }
        if (lang==null) {
            lang = Locale.getDefault().getLanguage();
        }
        return (String) largeIcons.get(lang);
    }
    
    /** 
     * @return a map of localized large icons uris indexed
     * by language
     */
    public Map getLocalizedLargeIconUris() {
        return largeIcons;
    }
    
    /**
     * set the localized small icon uri for the passed language
     * @param the language
     * @param the uri for the small icon
     */
    public void setLocalizedSmallIconUri(String lang, String uri) {
        if (lang==null) {
            lang = Locale.getDefault().getLanguage();
        }
        if (smallIcons==null) {
            smallIcons = new HashMap();
        }
        smallIcons.put(lang, uri);
        changed(LARGE_ICON_CHANGED);
    }
    
    /**
     * @return the small icon uri for the passed language
     */
    public String getLocalizedSmallIconUri(String lang) {
        if (smallIcons==null) {
            return null;
        }
        if (lang==null) {
            lang = Locale.getDefault().getLanguage();
        }
        return (String) smallIcons.get(lang);
    }
    
    /**
     * @return the map of small icons indexed by language
     */
    public Map getLocalizedSmallIconUris() {
        return smallIcons;
    }
    
    /**
    * The large icon name of this descriptor as a String.
    *
    *  @return the large icon name of this descriptor 
    *
    */
    public String getLargeIconUri() {
        return getLocalizedLargeIconUri(null);
    }
    
    /**
    * Sets the large icon name of this descriptor as a String.
    *
    *  @param largeIconUri the large icon name of this descriptor 
    *
    */
    public void setLargeIconUri(String largeIconUri) {
        setLocalizedLargeIconUri(null, largeIconUri);
    }
    
    /**
    * The small icon name of this descriptor as a String.
    *
    *  @return the small icon name of this descriptor 
    *
    */
    public String getSmallIconUri() {
        return getLocalizedSmallIconUri(null);
    }
    
    /**
    * Sets the small icon name of this descriptor as a String.
    *
    *  @param smallIconUri the small icon name of this descriptor 
    *
    */
    public void setSmallIconUri(String smallIconUri) {
        setLocalizedSmallIconUri(null, smallIconUri);
    }
    
    /**
    * Register the given notification listener for change events.
    *
    *  @param nl the notification listener to register 
    *
    */
    public void addNotificationListener(NotificationListener nl) {
	if ((nl != null) && !listeners.contains(nl)) { // avoid duplicate listeners
	    listeners.addElement(nl);
	}
    }
    
    /**
    * Deregister the given notification listener for change events.
    *
    *  @param nl the notification listener to deregister 
    *
    */
    public void removeNotificationListener(NotificationListener nl) {
	listeners.removeElement(nl);
    }

    /**
    * return current notification listeners
    *
    */
    public List getNotificationListeners() {
	Vector listenersClone = null;
	synchronized (listeners) {
	    listenersClone = (Vector)this.listeners.clone();
	}
	return listenersClone;
    }

    /**
    * Notify all my listeners that I have changed..
    *
    */
    public void changed(String attribute) {
	// fine-grain attribute currently ignored
	// eventually, 'changed(String)' should replace 'changed()' everywhere.
	this.changed();
    }

    /**
    * Notify all my listeners that I have changed..
    *
    */
    public void changed() {
	String attribute = "";
	NotificationEvent ne = new NotificationEvent(this, DESCRIPTOR_CHANGED, this, attribute);
	
	List listenersClone = this.getNotificationListeners();
	for (Iterator e = listenersClone.iterator(); e.hasNext();) {
	    NotificationListener nl = (NotificationListener)e.next();
	    nl.notification(ne);   
	}
    }
    
    /**
    * Returns the largest substring of the given string that
    * does not have an integer at the end.
    *
    */
    private static String stripIntegerEndingFrom(String s) {
	return recursiveStripIntegerEndingFrom(s);
    }
    
    /**
    * Returns the largest substring of the given string that
    * does not have an integer at the end.
    *
    */
    private static String recursiveStripIntegerEndingFrom(String s) {
	if (s.length() > 1) {
	    String shorterByOne = s.substring(0, s.length() - 1);
	    
	    String lastBit = s.substring(s.length() - 1, s.length());
	    try {
		Integer.parseInt(lastBit);
		return recursiveStripIntegerEndingFrom(shorterByOne);
	    } catch (NumberFormatException nfe) {
		return s;
	    }
	
	}
	return s;
    }
    
    /**
    * Returns String based on the trial name that is guaramteed to be different
    * from any of the strings in the vector of String names.
    *
    */
    private static String uniquifyString(String trialName, Vector v, int index) {
	for(Enumeration e = v.elements(); e.hasMoreElements();) {
	    String next = (String) e.nextElement();
	    if (next.equals(trialName)) {
		index++;
		return uniquifyString(stripIntegerEndingFrom(trialName) + index, v, index); 
	    }
	}
	return trialName;
    }
        
    /**
    * Returns String based on the trial name that is guaramteed to be different
    * from any of the strings in the vector of String names.
    * 
    * @param trialName the suggested name
    * @param otherNames The Vector of String objects none of which will be the same as the return
    * @return the unique String 
    */
    public static String createUniqueFilenameAmongst(String trialName, Vector otherNames) {

	/* extract file.ext */
	int p = trialName.lastIndexOf(".");
	if (p < 0) {
	    return uniquifyString(trialName, otherNames, 0);
	}
	String ext  = trialName.substring(p);
	String file = trialName.substring(0, p);

	/* get list of filenames less extension */
	Vector nameList = new Vector();
	for (Enumeration e = otherNames.elements(); e.hasMoreElements();) {
	    String name = e.nextElement().toString();
	    if (name.endsWith(ext)) {
	  	nameList.add(name.substring(0, name.length() - ext.length()));
	    }
	}
	String unique = uniquifyString(file, nameList, 0);
	return unique + ext;

    }

    /**
    * Returns String based on the trial name that is guaramteed to be different
    * from any of the strings in the vector of String names.
    * 
    * @param trialName the suggested name
    * @param otherNames The Vector of String objects none of which will be the same as the return
    * @return the unique String 
    */
    public static String createUniqueNameAmongst(String trialName, Vector otherNames) {
	return uniquifyString(trialName, otherNames, 0);
    }
    
    /**
    * Returns String based on the trial name that is guaramteed to be different
    * from any of the strings returnsed by the getName() call in any of the Descriptor objects in the Set supplied.
    * 
    * @param trialName the suggested name
    * @param descriptors The Set of Descriptor objects to whose name attribute will not be the same as the return
    * @return the unique String 
    */
    public static String createUniqueNameAmongstNamedDescriptors(String trialName, Set descriptors) {
	Vector v = new Vector();
	for (Iterator itr = descriptors.iterator(); itr.hasNext();) {
	    Descriptor next = (Descriptor) itr.next();
	    v.addElement(next.getName());
	}
	return createUniqueNameAmongst(trialName, v);
    }
    
    /**
     * Add a new deployment-extension for this descriptor
     * @param deployment-extension descriptor to add
     */
    public void addDeploymentExtension(DeploymentExtensionDescriptor de) {
        Vector extensions = (Vector) getExtraAttribute("deployment-extension");
        if (extensions==null) {
            extensions = new Vector();
            addExtraAttribute("deployment-extension", extensions);
        }
        extensions.add(de);
    }
    
    /**
     * @return an iterator on the deployment-extension
     */
    public Iterator getDeploymentExtensions() {
        Vector extensions = (Vector) getExtraAttribute("deployment-extension");
        if (extensions!=null) {
            return extensions.iterator();
        } 
        return null;
    }
    
    /**
     * add a prefix mapping
     */
    public void addPrefixMapping(String mapping, String uri) {
        Map prefixMapping = getPrefixMapping();
        if (prefixMapping==null) {
            prefixMapping = new java.util.HashMap();                        
            addExtraAttribute("prefix-mapping", prefixMapping);
        }
        prefixMapping.put(mapping, uri);
    }
    
    /**
     * @return the map of prefix to namepace uri
     */
    public Map getPrefixMapping() {
        return (Map) getExtraAttribute("prefix-mapping");
    }
    
    /**
    * A String representation of this object.
    */
    public void print(StringBuffer toStringBuffer) {
        StringBuffer sb = toStringBuffer;
        
        if (displayNames!=null) {
            sb.append("Display Names:");
            displayLocalizedMap(sb, displayNames);
        }
        if (descriptions!=null) {
            sb.append("\n Descriptions");
            displayLocalizedMap(sb, descriptions);
        }
        if (smallIcons!=null) {
            sb.append("\n SmallIcons");
            displayLocalizedMap(sb, smallIcons);
        }
        if (largeIcons!=null) {
            sb.append("\n LargeIcons");
            displayLocalizedMap(sb, largeIcons);
        }
        Map prefix = getPrefixMapping();
        if (prefix!=null) 
            sb.append("\n Prefix Mapping = ").append(prefix);        
        Iterator itr = getDeploymentExtensions();
        if (itr!=null && itr.hasNext()) {
            do {
                sb.append("\n Deployment Extension : ");
                ((Descriptor)(itr.next())).print(sb);
            } while (itr.hasNext());
        }
	sb.append("\n");
        super.print(sb);
    }
    
    /** 
     * helper method to display a localized map
     */
    private void displayLocalizedMap(StringBuffer sb, Map localizedMap) {
        for (Iterator itr = localizedMap.keySet().iterator();itr.hasNext();) {
            String lang = (String) itr.next();
            sb.append("\n   lang[" + lang + "]  = " + localizedMap.get(lang));
        }
    }
        
    
    /**
     * Visitor API implementation, all descriptors must be visitable
     * 
     * @param  the visitor implementation
     */
    public void visit(DescriptorVisitor aVisitor) {
        aVisitor.accept(this);
    }     
    
    /**
     * @return the default locale language
     */
    private String getDefaultLanguage() {
        return Locale.getDefault().getLanguage();
    }
    
    //start IASRI 4713550
    public String docType = null;
    
    public String getDocType(){
        return docType;
    }   
    
    public void fillDocType(InputStream in){
        try{
            BufferedReader inr = new BufferedReader(new InputStreamReader(in));
            String s = inr.readLine();
            while (s != null) {
                if (s.indexOf("DOCTYPE") > -1) {
                    docType = s;
                    in.close();
                    return;
                }   
                s = inr.readLine();
            }   
        }catch(Exception e){
        }
    }   
    //end IASRI 4713550    
}
