/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment.annotation.handlers;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Method;
import java.lang.NoSuchMethodException;
import java.util.logging.Level;

import javax.ejb.Init;

import com.sun.enterprise.deployment.EjbDescriptor;
import com.sun.enterprise.deployment.EjbInitInfo;
import com.sun.enterprise.deployment.EjbSessionDescriptor;
import com.sun.enterprise.deployment.MethodDescriptor;
import com.sun.enterprise.util.TypeUtil;

import com.sun.enterprise.deployment.annotation.AnnotationInfo;
import com.sun.enterprise.deployment.annotation.AnnotationProcessorException;
import com.sun.enterprise.deployment.annotation.HandlerProcessingResult;
import com.sun.enterprise.deployment.annotation.context.EjbContext;

/**
 * This handler is responsible for handling the javax.ejb.Init attribute
 *
 */
public class InitHandler extends AbstractAttributeHandler {
    
    public InitHandler() {
    }
    
    /**
     * @return the annoation type this annotation handler is handling
     */
    public Class<? extends Annotation> getAnnotationType() {
        return Init.class;
    }    
        
    protected HandlerProcessingResult processAnnotation(AnnotationInfo ainfo,
            EjbContext[] ejbContexts) throws AnnotationProcessorException {

        Init init = (Init) ainfo.getAnnotation();

        for(EjbContext next : ejbContexts) {
            
            EjbSessionDescriptor sessionDescriptor = 
                (EjbSessionDescriptor) next.getDescriptor();

            Method m = (Method) ainfo.getAnnotatedElement();

            // Check for matching method on home and/or local home interface.

            int numMatches = 0;

            
            String adaptedCreateMethodName = init.value();

            try {
                if( sessionDescriptor.isRemoteInterfacesSupported() ) {
                    addInitMethod(sessionDescriptor, m, 
                                  adaptedCreateMethodName, false);
                    numMatches++;
                }
            } catch(Exception e) {
            }                   

            try {                
                if( sessionDescriptor.isLocalInterfacesSupported() ) {
                    addInitMethod(sessionDescriptor, m, 
                                  adaptedCreateMethodName, true);
                    numMatches++;
                }
            } catch(Exception e) {                
            }

            if( numMatches == 0 ) {
                log(Level.SEVERE, ainfo, 
                    localStrings.getLocalString(
                    "enterprise.deployment.annotation.handlers.notmatchcreate",
                    "Unable to find matching Home create method for Init method {0} on bean {1}.",
                    new Object[] { m, sessionDescriptor.getName() }));
                return getDefaultFailedResult();
            }
        }

        return getDefaultProcessedResult();
    }

    private void addInitMethod(EjbSessionDescriptor descriptor,
                               Method beanMethod, 
                               String adaptedCreateMethodName, boolean local) 
        throws Exception {

        String homeIntfName = local ? descriptor.getLocalHomeClassName() :
            descriptor.getHomeClassName();            

        ClassLoader cl = descriptor.getEjbBundleDescriptor().getClassLoader();

        Class homeIntf = cl.loadClass(homeIntfName);
        
        Method createMethod = null;
        if( (adaptedCreateMethodName == null) || 
            (adaptedCreateMethodName.equals("")) ) {
            // Can't make any assumptions about matching method name.  Could
            // be "create" or some form of create<METHOD>, so match based on
            // signature.
            for(Method next : homeIntf.getMethods()) {
                if( next.getName().startsWith("create") &&
                    TypeUtil.sameParamTypes(next, beanMethod) ) {
                    createMethod = next;
                    break;
                }
            }
            if( createMethod == null ) {
                throw new NoSuchMethodException("No matching adapted home " +
                                                "method found for @Init " +
                                                " method " + beanMethod);
            }
        } else {
            createMethod = homeIntf.getMethod(adaptedCreateMethodName, 
                                              beanMethod.getParameterTypes());
        }
        
        MethodDescriptor beanMethodDescriptor =
            new MethodDescriptor(beanMethod, MethodDescriptor.EJB_BEAN);
        
        MethodDescriptor createMethodDescriptor = 
            new MethodDescriptor(createMethod, 
                                 ( local ?
                                   MethodDescriptor.EJB_HOME :
                                   MethodDescriptor.EJB_LOCALHOME ));
        
        EjbInitInfo initInfo = new EjbInitInfo();
                    
        initInfo.setBeanMethod(beanMethodDescriptor);
        initInfo.setCreateMethod(createMethodDescriptor);

        descriptor.addInitMethod(initInfo);
        
    }

    /**
     * @return an array of annotation types this annotation handler would 
     * require to be processed (if present) before it processes it's own 
     * annotation type.
     */
    public Class<? extends Annotation>[] getTypeDependencies() {
        return getEjbAnnotationTypes();
    }
}
