/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.ejb.containers;

import java.util.*;

import javax.ejb.*;
import javax.transaction.*;

import com.sun.enterprise.log.Log;

import java.util.logging.*;
import com.sun.logging.*;


/**
 * This class intercepts Synchronization notifications from
 * the TransactionManager and forwards them to the appropriate container.
 * There is one ContainerSynchronization instance per tx.
 * All bean instances (of all types) participating in the tx must be
 * added by the container to the ContainerSynchronization, so that
 * the beans can be called during before/afterCompletion.
 *
 * This class also provides special methods for PersistenceManager Sync and
 * Timer objects which must be called AFTER the containers during 
 * before/afterCompletion.
 *
 */
final class ContainerSynchronization implements Synchronization
{

    private static final Logger _logger =
        LogDomains.getLogger(LogDomains.EJB_LOGGER);

    private Vector beans = new Vector();
    private Vector pmSyncs = new Vector();

    private Hashtable timerSyncs = new Hashtable();

    private Transaction tx; // the tx with which this Sync was registered
    private ContainerFactoryImpl containerFactory;
    private SFSBTxCheckpointCoordinator sfsbTxCoordinator;

    // Note: this must be called only after a Tx is begun.
    ContainerSynchronization(Transaction tx, 
			     ContainerFactoryImpl containerFactory)
    {
        this.tx = tx;
        this.containerFactory = containerFactory;
    }
    
    Vector  getBeanList(){
        return beans;
    }
    
    void addBean(EJBContextImpl bean)
    {
        if (beans.contains(bean))
            return;
        beans.add(bean);
    }
    
    void removeBean(EJBContextImpl bean)
    {
        beans.remove(bean);
    }
    
    void addPMSynchronization(Synchronization sync)
    {
        pmSyncs.add(sync);
    }

    // Set synchronization object for a particular timer.  
    void addTimerSynchronization(TimerPrimaryKey timerId, Synchronization sync)
    {
        timerSyncs.put(timerId, sync);
    }

    // Might be null if no timer synch object for this timerId in this tx
    Synchronization getTimerSynchronization(TimerPrimaryKey timerId) {
        return (Synchronization) timerSyncs.get(timerId);
    }

    void removeTimerSynchronization(TimerPrimaryKey timerId) {
        timerSyncs.remove(timerId);
    }

    public void beforeCompletion()
    {
        // first call beforeCompletion for each bean instance
        for ( int i=0; i<beans.size(); i++ ) {
            EJBContextImpl context = (EJBContextImpl)beans.elementAt(i);
            BaseContainer container = (BaseContainer)context.getContainer();
            try {
                if( container != null ) {
		    if (container.isUndeployed()) {
			_logger.log(Level.WARNING, "Marking Tx for rollback "
			    + " because container for " + container
			    + " is undeployed");
			try {
			    tx.setRollbackOnly();
			} catch (SystemException sysEx) {
			    _logger.log(Level.FINE, "Error while trying to "
				+ "mark for rollback", sysEx);
			}
		    } else {
			container.beforeCompletion(context);
		    }
                } else {
                    // Might be null if bean was removed.  Just skip it.
                    _logger.log(Level.FINE, "context with empty container in " +
                                " ContainerSynchronization.beforeCompletion");
                }
            } catch ( Exception ex ) {
                // rollback the Tx. The client will get
                // a EJB/RemoteException or a TransactionRolledbackException.
                _logger.log(Level.SEVERE,"ejb.remote_or_txnrollback_exception",ex);
                try {
                    tx.setRollbackOnly();
                } catch ( SystemException e ) {
                    _logger.log(Level.FINE, "", ex);
                }
                return; // no need to call remaining beforeCompletions
            }
        }

        // now call beforeCompletion for all pmSyncs
        for ( int i=0; i<pmSyncs.size(); i++ ) {
            Synchronization sync = (Synchronization)pmSyncs.elementAt(i);
            try {
                sync.beforeCompletion();
            } catch ( Exception ex ) {
                // rollback the Tx. The client will get
                // a EJB/RemoteException or a TransactionRolledbackException.
                _logger.log(Level.SEVERE,"ejb.remote_or_txnrollback_exception",ex);
                try {
                    tx.setRollbackOnly();
                } catch ( SystemException e ) {
                    _logger.log(Level.FINE, "", ex);
                }
                return; // no need to call remaining beforeCompletions
            }
        }
    }

    public void afterCompletion(int status)
    {	
        for ( int i=0; i<pmSyncs.size(); i++ ) {
            Synchronization sync = (Synchronization)pmSyncs.elementAt(i);
            try {
                sync.afterCompletion(status);
            } catch ( Exception ex ) {
                _logger.log(Level.SEVERE, "ejb.after_completion_error", ex);
            }
        }

        // call afterCompletion for each bean instance
        for ( int i=0; i<beans.size();i++  ) {
            EJBContextImpl context = (EJBContextImpl)beans.elementAt(i);
            BaseContainer container = (BaseContainer)context.getContainer();
            try {
                if( container != null ) {
                    container.afterCompletion(context, status);
                } else {
                    // Might be null if bean was removed.  Just skip it.
                    _logger.log(Level.FINE, "context with empty container in "
                                +
                                " ContainerSynchronization.afterCompletion");
                }
            } catch ( Exception ex ) {
                _logger.log(Level.SEVERE, "ejb.after_completion_error", ex);
            }
        }

	if (sfsbTxCoordinator != null) {
	    sfsbTxCoordinator.doTxCheckpoint();
	}

        for ( Iterator iter = timerSyncs.values().iterator(); 
              iter.hasNext(); ) {
            Synchronization timerSync = (Synchronization) iter.next();
            try {
                timerSync.afterCompletion(status);
            } catch ( Exception ex ) { 
                _logger.log(Level.SEVERE, "ejb.after_completion_error", ex);
            }
        }

        // tell ContainerFactory to remove this tx/sync from its table
        containerFactory.removeContainerSync(tx);
    }

    void registerForTxCheckpoint(SessionContextImpl sessionCtx) {
	//No need to synchronize
	if (sfsbTxCoordinator == null) {
	    sfsbTxCoordinator = new SFSBTxCheckpointCoordinator();
	}

	sfsbTxCoordinator.registerContext(sessionCtx);
    }
}
