/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.ejb.containers;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Proxy;
import java.lang.reflect.Method;

import java.util.logging.Logger;
import java.util.logging.Level;

import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

import java.rmi.RemoteException;
import javax.rmi.CORBA.Tie;
import javax.ejb.EJBException;
import javax.ejb.EJBHome;
import com.sun.ejb.Invocation;
import com.sun.ejb.InvocationInfo;
import com.sun.ejb.Container;
import com.sun.ejb.containers.util.MethodMap;
import com.sun.enterprise.deployment.EjbDescriptor;
import com.sun.enterprise.deployment.EjbSessionDescriptor;
import com.sun.enterprise.util.LocalStringManagerImpl;
import com.sun.enterprise.util.Utility;

import com.sun.logging.LogDomains;

/** 
 * Handler for EJBHome invocations through EJBHome proxy.
 *
 * @author Kenneth Saks
 */    

final class EJBHomeInvocationHandler 
    extends ReadOnlyEJBHomeImpl implements InvocationHandler {

    private static final Logger logger =
        LogDomains.getLogger(LogDomains.EJB_LOGGER);

    private static LocalStringManagerImpl localStrings =
        new LocalStringManagerImpl(EJBHomeInvocationHandler.class);

    private boolean isStatelessSession_;
    private boolean isEntity_;

    // Our associated proxy object.  Used when a caller needs EJBHome
    // but only has InvocationHandler.
    private EJBHome proxy_;

    private Class homeIntfClass_;

    // Cache reference to invocation info.  There is one of these per
    // container.  It's populated during container initialization and
    // passed in when the InvocationHandler is created.  This avoids the
    // overhead of building the method info each time a Home proxy
    // is created.  
    private MethodMap invocationInfoMap_;

    EJBHomeInvocationHandler(EjbDescriptor ejbDescriptor,
                             Class homeIntfClass,
                             MethodMap invocationInfoMap) 
        throws Exception {

        if( ejbDescriptor instanceof EjbSessionDescriptor ) {
            isEntity_ = false;
            isStatelessSession_ = 
                ((EjbSessionDescriptor)ejbDescriptor).isStateless();
        } else {
            isStatelessSession_ = false;
            isEntity_ = true;
        }

        invocationInfoMap_ = invocationInfoMap;

        homeIntfClass_ = homeIntfClass;

        // NOTE : Container is not set on super-class until after 
        // constructor is called.
    }

    public void setProxy(EJBHome proxy) {
        proxy_ = proxy;
    }

    protected EJBHome getEJBHome() {
        return proxy_;
    }

    /**
     * Called by EJBHome proxy.
     */
    public Object invoke(Object proxy, Method method, Object[] args) 
        throws Throwable {

        ClassLoader originalClassLoader = null;

        // NOTE : be careful with "args" parameter.  It is null
        //        if method signature has 0 arguments.
        try {
            ((BaseContainer) getContainer()).onEnteringContainer();

            // In some cases(e.g. if the Home/Remote interfaces appear in
            // a parent of the application's classloader), 
            // ServantLocator.preinvoke() will not be called by the
            // ORB, and therefore BaseContainer.externalPreInvoke will not have
            // been called for this invocation.  In those cases we need to set 
            // the context classloader to the application's classloader before 
            // proceeding. Otherwise, the context classloader could still 
            // reflect the caller's class loader.  
            
            if( Thread.currentThread().getContextClassLoader() != 
                getContainer().getClassLoader() ) {
                originalClassLoader = Utility.setContextClassLoader
                    (getContainer().getClassLoader());
            }

            Class methodClass = method.getDeclaringClass();

            if( methodClass == java.lang.Object.class )  {
                return InvocationHandlerUtil.invokeJavaObjectMethod
                    (this, method, args);    
            } else if( methodClass == ReadOnlyEJBHome.class ) {
                if( method.getName().equals("_refresh_All") ) {
                    super._refresh_All();
                } else {
                    super._refresh_com_sun_ejb_containers_read_only_bean_
                        (args[0]);
                }
                return null;
            }

            // Use optimized version of get that takes param count as an 
            // argument.
            InvocationInfo invInfo = (InvocationInfo)
                invocationInfoMap_.get(method, 
                                       ((args != null) ? args.length : 0) );
            
            if( invInfo == null ) {

                throw new RemoteException("Unknown Home interface method :" 
                                          + method);

            } else if( (methodClass == javax.ejb.EJBHome.class) ||
                       invInfo.ejbIntfOverride ) {
                
                return invokeEJBHomeMethod(method.getName(), args);
                
            } else if( GenericEJBHome.class.isAssignableFrom(methodClass) ) {
                
                // This is an internal creation request through the EJB 3.0
                // client view, so just create an business object and return it
                EJBObjectImpl busObjectImpl = createRemoteBusinessObjectImpl();
                return busObjectImpl.getStub((String) args[0]);
                
            }                
            
            // Process finder, create method, or home method.
            EJBObjectImpl ejbObjectImpl = null;
            Object returnValue = null;
            
            if( !isEntity_ && invInfo.startsWithCreate ) {
                ejbObjectImpl = createEJBObjectImpl();
                returnValue = ejbObjectImpl.getStub();
            }
            
            if( !isStatelessSession_ ) {
                
                if( invInfo.targetMethod1 == null ) {
                    
                    String errorMsg = localStrings.getLocalString
                        ("ejb.bean_class_method_not_found", "", new Object[]
                            { invInfo.ejbName, "Home", 
                              invInfo.method.toString() });
                    logger.log(Level.SEVERE, errorMsg);
                    throw new RemoteException(errorMsg);            
                    
                }
                
                Invocation inv = new Invocation();
                
                inv.isLocal = false;
                inv.method  = method;
                inv.isHome  = true;
                
                inv.clientInterface = homeIntfClass_;

                // Set cached invocation params.  This will save 
                // additional lookups in BaseContainer.

                inv.transactionAttribute = invInfo.txAttr;
                inv.securityPermissions = invInfo.securityPermissions;
                inv.invocationInfo = invInfo;
                
                if( !isEntity_ && invInfo.startsWithCreate ) {
                    inv.ejbObject = (EJBLocalRemoteObject) ejbObjectImpl;
                }
                
                BaseContainer container = (BaseContainer) getContainer();
                
                try {
                    
                    container.preInvoke(inv);
                    
                    if( invInfo.startsWithCreate ) {
                        
                        Object ejbCreateReturnValue = container.
                            invokeTargetBeanMethod(invInfo.targetMethod1, 
                                                   inv, inv.ejb, args, null);
                        if( isEntity_ ) {
                            container.postCreate(inv, ejbCreateReturnValue);
                            container.invokeTargetBeanMethod
                                (invInfo.targetMethod2, 
                                 inv, inv.ejb, args, null);
                        } 
                        if( inv.ejbObject != null ) {
                            returnValue = ((EJBObjectImpl)inv.ejbObject).
                                getStub();
                        } 
                        
                    } else if (invInfo.startsWithFindByPrimaryKey) {
                        EntityContainer entityContainer = (EntityContainer) 
                            container;
                        returnValue = entityContainer.invokeFindByPrimaryKey
                            (invInfo.targetMethod1, inv, args);
                                                                             
                    } else if ( invInfo.startsWithFind ) {
                        
                        Object pKeys = container.invokeTargetBeanMethod
                            (invInfo.targetMethod1, inv, inv.ejb, args, null);
                                                                        
                        returnValue = container.postFind(inv, pKeys, null);
                        
                    } else {
                        
                        returnValue = container.invokeTargetBeanMethod
                            (invInfo.targetMethod1, inv, inv.ejb, args, null);
                    }
                } catch(InvocationTargetException ite) {
                    inv.exception = ite.getCause();
                } catch(Throwable c) {
                    inv.exception = c;
                } finally {
                    container.postInvoke(inv);
                }
                
                if (inv.exception != null) {
                    InvocationHandlerUtil.throwRemoteException
                        (inv.exception, method.getExceptionTypes());
                }
            }
            
            return returnValue;
        } finally {
            
            if( originalClassLoader != null ) {
                Utility.setContextClassLoader(originalClassLoader);
            }

            ((BaseContainer) getContainer()).onLeavingContainer();
        }
    }

    private Object invokeEJBHomeMethod(String methodName, Object[] args)
        throws Exception
    {
        // Return value is null if target method returns void.
        Object returnValue = null;

        // NOTE : Might be worth optimizing this method check if it
        // turns out to be a bottleneck.  I don't think these methods
        // are called with the frequency that this would be an issue,
        // but it's worth considering.

        if( methodName.equals("getEJBMetaData") ) {

            returnValue = super.getEJBMetaData();

        } else if( methodName.equals("getHomeHandle") ) {

            returnValue = super.getHomeHandle();

        } else if( methodName.equals("remove") ) {

            if( args[0] instanceof javax.ejb.Handle ) {
                super.remove((javax.ejb.Handle)args[0]);
            } else {
                super.remove(args[0]);
            }

        } else {

           throw new RemoteException("unknown EJBHome method = " + methodName);

        }

        return returnValue;
    }
}
