/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.admin.server.core;

import java.util.logging.Level;
import java.util.logging.Logger;

import java.net.URL;

import javax.management.MBeanServer;

import com.sun.enterprise.admin.AdminContext;
import com.sun.enterprise.admin.meta.MBeanRegistry;
import com.sun.enterprise.admin.meta.MBeanRegistryFactory;
import com.sun.enterprise.admin.meta.AdminConfigEventListener;
import com.sun.enterprise.admin.util.proxy.Interceptor;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigContextEventListener;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.ConfigContextFactory;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.server.ServerContextImpl;
import com.sun.enterprise.config.pluggable.ConfigEnvironment;
import com.sun.enterprise.config.pluggable.EnvironmentFactory;
import com.sun.enterprise.config.serverbeans.ServerHelper;

/**
 *
 */
public class AdminContextImpl implements AdminContext {

    private ServerContextImpl serverContext;
    private ConfigContext runtimeConfigContext;
    private ConfigContext adminConfigContext;
    private MBeanServer mbeanServer;
    private String domainName;
    private String serverName;
    private Logger adminLogger;
    private Interceptor configInterceptor;
    private URL    adminMBeanRegistryURL;
    private URL    runtimeMBeanRegistryURL;
    
    private static final String ADMIN_DESCRIPTORS_FILENAME   = "/admin-mbeans-descriptors.xml";
    private static final String RUNTIME_DESCRIPTORS_FILENAME = "/runtime-mbeans-descriptors.xml";

    /**
     * Creates a new instance of AdminContextImpl
     */
    public AdminContextImpl() {
        String loggerName = System.getProperty("com.sun.aas.admin.logger.name");
        if (loggerName != null) {
            adminLogger = Logger.getLogger(loggerName);
        } else {
            adminLogger = Logger.getAnonymousLogger();
        }
        //default PE settings 
        domainName = "com.sun.appserv";
        //notify MBeanRegistryFactory
        MBeanRegistryFactory.setAdminContext(this);
    }

    public AdminContextImpl(ServerContextImpl srvCtx) {
        this();
        setServerContext(srvCtx);
    }


    public void setServerContext(ServerContextImpl srvCtx) {
        serverContext = srvCtx;
        runtimeConfigContext = serverContext.getConfigContext();
        String configFileName = serverContext.getServerConfigURL();
        try {
            adminConfigContext = ConfigContextFactory.createConfigContext(
                    getAdminConfigEnvironment(configFileName));
            // Registering the config validator
            if (ServerHelper.isDAS(srvCtx.getConfigContext(), srvCtx.getInstanceName())) {
                registerValidator();
                registerConfigAdjustmentListener();
            }
        } catch (ConfigException ce) {
            adminLogger.log(Level.SEVERE, "core.determining_server_instance_failed",
                   ce);
        } catch (RuntimeException ce) {
            adminLogger.log(Level.SEVERE, "core.admin_config_read_error",
                   ce.getMessage());
            adminLogger.log(Level.WARNING, "core.admin_config_read_error_trace",
                   ce);
            throw ce;
        }
              
        adminLogger.log(Level.FINEST, "core.log_config_id_runtime",
                new Long(runtimeConfigContext.hashCode()));
        adminLogger.log(Level.FINEST, "core.log_config_is_admin",
                new Long(adminConfigContext.hashCode()));
        serverName = serverContext.getInstanceName();
    }

    protected void registerConfigAdjustmentListener() {
	
        // Added mechanism to avoid unadjusted domain.xml.<->MBeans<->dotted_names
             adminConfigContext.addConfigContextEventListener((ConfigContextEventListener)(new AdminConfigEventListener()));
    }
    protected void registerValidator() {
	
        // Added reflection mechanism to avoid compile time error
	try {
             Class cl = Class.forName("com.sun.enterprise.config.serverbeans.validation.DomainMgr");
             adminConfigContext.addConfigContextEventListener((ConfigContextEventListener)cl.newInstance());
        } catch (Exception ex) {
             adminLogger.log(Level.WARNING, "core.admin_validator_not_registered", "Error registering validator, config validator will not be available");
             adminLogger.log(Level.FINE, "core.admin_validator_register_error", ex.getMessage());
        }
    }

    public ConfigContext getAdminConfigContext() {
        return adminConfigContext;
    }
   
    public String getDomainName() {
        return domainName;
    }

    public MBeanServer getMBeanServer() {
        return mbeanServer;
    }

    public ConfigContext getRuntimeConfigContext() {
        return runtimeConfigContext;
    }
    
    public String getServerName() {
        return serverName;
    }
    
    public URL getAdminMBeanRegistryURL() {
        if(adminMBeanRegistryURL==null) {
            try { 
                //PE standard 
                adminMBeanRegistryURL  = MBeanRegistry.class.getResource(ADMIN_DESCRIPTORS_FILENAME);
            } catch (Throwable t) {
                adminLogger.log(Level.WARNING, "core.adminconfig_adminregistry_not_found",
                       ADMIN_DESCRIPTORS_FILENAME);
            }
        }
        return adminMBeanRegistryURL;
    }
    
    public URL getRuntimeMBeanRegistryURL() {
        if(runtimeMBeanRegistryURL==null) {
            try { 
                //PE standard 
                runtimeMBeanRegistryURL  = MBeanRegistry.class.getResource(RUNTIME_DESCRIPTORS_FILENAME);
            } catch (Throwable t) {
                adminLogger.log(Level.WARNING, "core.adminconfig_runtimeregistry_not_found",
                       RUNTIME_DESCRIPTORS_FILENAME);
            }
        }
        return runtimeMBeanRegistryURL;
    }
    
    public void setAdminConfigContext(ConfigContext ctx) {
    }
    
    public void setDomainName(String name) {
//        domainName = name;
    }
    
    public void setMBeanServer(MBeanServer mbs) {
        mbeanServer = mbs;
    }

    public void setRuntimeConfigContext(ConfigContext ctx) {
        if (serverContext != null) {
            serverContext.setConfigContext(ctx);
        }
    }
    
    public void setServerName(String serverName) {
    }

    public void setAdminMBeanRegistryURL(URL url) {
        adminMBeanRegistryURL = url;
    }
    
    public void setRuntimeMBeanRegistryURL(URL url) {
        runtimeMBeanRegistryURL = url;
    }
    
    public Logger getAdminLogger() {
        return adminLogger;
    }

    public void setAdminLogger(Logger logger) {
        adminLogger = logger;
    }

    public Interceptor getMBeanServerInterceptor() {
        if (configInterceptor == null) {
            configInterceptor = new ConfigInterceptor(this);
        }
        return configInterceptor;
    }

    public void setMBeanServerInterceptor(Interceptor interceptor) {
    }

    private ConfigEnvironment getAdminConfigEnvironment(String configFileName) {
        ConfigEnvironment ce = EnvironmentFactory.getEnvironmentFactory().
                                getConfigEnvironment();
        ce.setUrl(configFileName);
        ce.setReadOnly(false);
        ce.setCachingEnabled(false);
        ce.setRootClass("com.sun.enterprise.config.serverbeans.Domain");
        ce.setHandler("com.sun.enterprise.config.serverbeans.ServerValidationHandler");
        ce.getConfigBeanInterceptor().setResolvingPaths(false);
        return ce;
    }
    public String getDottedNameMBeanImplClassName(){
        return  "com.sun.enterprise.admin.mbeans.DottedNameGetSetMBeanImpl";
    }
}
