/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.server.core.jmx;

//JDK imports
import java.util.Set;

//JMX imports
import javax.management.ObjectName;

/**
    Defines the interface for a Repository of MBeans. Note that there is a 
    one to many relationship between an MBeanServer and Domains of MBeans.
    This interface abstracts out that relationship and hides the fact
    that in fact there can be multiple Repositories, one per Domain.
    @author Kedar Mhaswade
    @version 1.0
*/

public interface IRepository
{
    /**
		Adds an Object in the repository. Neither Object nor the ObjectName
		should be null. Given Object will not be added if an Object with the
		same ObjectName already exists in the repository.

		@param objectName - ObjectName of the Object to be stored.
		@param mbeanImpl - Object that is to be stored.

		@return true if the addition was successful, false otherwise.
    */
    public boolean add(ObjectName name, Object mbeanImpl);
    
    /**
		Removes the Object with given ObjectName from repository. The passed
		ObjectName must not be null.

		@param objectName - ObjectName of the Object to be removed.
		@return true if the Object could be removed, false otherwise.
    */
    public boolean remove(ObjectName objectName);
    
    /**
		Tests whether an Object with given ObjectName is present in this
		Repository.
		@param objectName - ObjectName of the Object to be searched for.
		@return true if the Object could be found, false otherwise.
		*/
    public boolean contains(ObjectName objectName);
    
    /**
		Returns the Object with given ObjectName. Returns null if there
		is no Object with given ObjectName. Passed ObjectName may not be null.
		@param objectName - ObjectName of the Object to be searched for.
		Note that ObjectName may not be a pattern for query or pattern
		on key properties.

		@return Object searched, null if there is none. Also returns null,
		if the ObjectName is pattern.
    */
    public Object find(ObjectName objectName);
    
    /**
        Makes the check for the existence of corresponding element in the 
        persistent store. This method will also register or unregister
        the MBeans as required by adding/removing them, depending on its existence
        in the persistent store.
        @return MBean that is registered, null otherwise. In case of the MBean
        that exists in both MBeanRepository and persistent store simply the
        object in memory is returned.
    */
    public Object findPersistent(ObjectName objectName);
    
    /**
		Returns the number of Objects stored in the repository for
		given domainName.
		@param domainName - String representing the name of domain.
    */
    public int getCount(String domainName);
	
	/**
		Returns the total number of MBeans stored in entire repository.
		
		@return int representing all MBeans in this repository.
	*/
	public int getTotalCount();
	
	/**
		Method to get ALL MBeans in all domains in this repository.
		The returned Set contains the ObjectNames of all MBeans.
	 
		@return Set containing MBean ObjectNames, null if none exists.
	*/
	public Set getAllMBeans();
	
	/**
		Returns a Set of Objects whose ObjectNames match zero or more Objects in the
		repository as per the pattern suggested by the argument. If the argument
		is NOT a pattern an exact match will be performed.
	 
		@param objectName the ObjectName that may represent a pattern.
		
		@return Set of all Objects that match the pattern in argument, null if
		there is no such object.
	*/
	public Set query(ObjectName objectName);
}