/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.appclient;

import java.text.MessageFormat;

/**
 *Represents any user error, such as an invalid combination of command options,
 *or specifying a non-existent JAR file.
 *<p>
 *Such errors should be user-correctable, provided the message is clear.
 *So no stack traces should be displayed when UserErrors are thrown.
 *
 * @author tjquinn
 */
public class UserError extends Throwable {

    /** Allows user to turn on stack traces for user errors - normally off */
    private static final String SHOW_STACK_TRACES_PROPERTY_NAME = "com.sun.aas.appclient.showUserErrorStackTraces";
    
    /**
     * Creates a new UserError instance having formatted the message with the 
     * arguments provided.
     *@param message the message string, presumably containing argument placeholders
     *@param args 0 or more arguments for substitution for the placeholders in the message string
     *@return new UserError with message formatted as requested
     */
    public static UserError formatUserError(String message, String... args) {
        String formattedMessage = MessageFormat.format(message, (Object[]) args);
        UserError ue = new UserError(formattedMessage);
        return ue;
    }

    /** xmlMessage implementation showed the usage message after the error */
    private String usage = null;

    public UserError(String message) {
        super(message);
    }

    public UserError(String message, Throwable cause) {
        super(message, cause);
    }

    public UserError(Throwable cause) {
        super(cause);
    }

    /**
     *Sets whether or not the usage message should be displayed after the
     *error message is displayed to the user.
     *@param showUsage the new setting 
     */
    public void setUsage(String usage) {
        this.usage = usage;
    }

    /**
     *Displays the user error message, and any messages along the exception
     *chain, if any, and then exits.  If showUsage has been set to true, then
     *the usage message is displayed before exiting.
     *<p>
     *Only the messages, and not the stack traces, are shown because these are
     *user errors that should be user-correctable.  Stack traces are too
     *alarming and of minimal use to the user as he or she tries to understand
     *and fix the error.
     */
    public void displayAndExit() {
        for (Throwable t = this; t != null; t = t.getCause()) {
            System.err.println(t.getLocalizedMessage());
        }
        if (usage != null) {
            System.err.println(usage);
        }
        if (Boolean.getBoolean(SHOW_STACK_TRACES_PROPERTY_NAME)) {
            printStackTrace();
        }
        System.exit(1);
    }

}
