/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.appclient.jws;

import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.interfaces.DeploymentImplConstants;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.instance.BaseManager;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;

/**
 *Represents an origin of user-provided content - application origins or
 *app client origins.
 *
 * @author tjquinn
 */
public abstract class UserContentOrigin extends ContentOrigin {

    /** descriptor for the origin */
    protected Application application;
    
    /**
     *Creates a new instance of the content origin.
     *@param the Application object for the content origin
     */
    public UserContentOrigin(Application application) {
        super();
        this.application = application;
    }

    /**
     *Returns the registration name for the top-level module associated with this
     *nested app client.
     *<p>
     *This method is primarily used to get a name to use in checking whether 
     *the relevant module has been enabled or disabled for Java Web Start access.
     *@return the application's registration name
     */
    public String getTopLevelRegistrationName() {
        return getApplication().getRegistrationName();
    }

    /**
     *Returns a string representation of this content origin.
     *@return string describing the origin's current state
     */
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append(" registration name=").append(application.getRegistrationName());
//                .append(", enabled=").append(enabled);
        return sb.toString();
    }
   
    /**
     *Returns the origin's corresponding Application object.
     *@return Application descriptor from which this origin was derived
     */
    public Application getApplication() {
        return application;
    }
    
    /**
     *Returns a File object for the generated app client jar file for this top-level
     *app client.
     *@return File for the generated jar file
     */
    public File locateGeneratedAppclientJarFile(BaseManager mgr) throws FileNotFoundException {
        String regName = application.getRegistrationName();
        
        String generatedAppClientJarDirSpec = mgr.getGeneratedXMLLocation(regName);
        String generatedAppClientJarName = getTopLevelRegistrationName() + DeploymentImplConstants.ClientJarSuffix;
        
        File generatedAppClientJar = new File (generatedAppClientJarDirSpec, generatedAppClientJarName);
        if ( ! generatedAppClientJar.exists()) {
            throw new FileNotFoundException(generatedAppClientJar.getAbsolutePath());
        }
        return generatedAppClientJar;        
    }
    /**
     *Returns whether this content origin's appclient is enabled for Java
     *Web Start access.
     *@return boolean indicating whether the appclient is enabled for JWS access
     */
    public boolean isEnabled() {
        try {
            return AppclientJWSSupportInfo.getInstance().isEnabled(this);
        } catch (IOException ioe) {
            /*
             *An IOException can occur if some infrastructure objects cannot be
             *located in obtaining the instance.  Very unlikely and also logged elsewhere.
             */
            return false;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}