/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.connectors;

import com.sun.enterprise.config.serverbeans.*;
import com.sun.enterprise.deployment.ConnectorDescriptor;
import com.sun.logging.LogDomains;
import javax.resource.spi.ManagedConnectionFactory;
import java.util.HashMap;
import java.util.logging.*;
import java.util.List;
import java.util.Map;
import java.util.ArrayList;
import com.sun.enterprise.connectors.authentication.RuntimeSecurityMap;


/** 
  * This is an registry class which holds various objects in hashMaps,
  * hash tables, and vectors. These objects are updated and queried 
  * during various funtionalities of rar deployment/undeployment, resource 
  * creation/destruction
  * Ex. of these objects are ResourcesAdapter instances, security maps for 
  * pool managed connection factories.
  * It follows singleton patter. i.e only one instance at any point of time.
  * @author    Binod P.G and Srikanth P
  * @version 
 */

public class ConnectorRegistry {

    protected static ConnectorRegistry connectorRegistryInstance;

    /**
     * <code>resourceAdapters</code> keeps track of all active resource
     * adapters in the connector runtime.
     * String:resourceadapterName Vs ActiveResourceAdapter
     */
    protected HashMap resourceAdapters;
    protected HashMap factories;
    protected HashMap resourceAdapterConfig;
    static Logger _logger = LogDomains.getLogger(LogDomains.RSR_LOGGER);

    /** Return the ConnectorRegistry instance
     *  @return ConnectorRegistry instance which is a singleton
     */
    
    public static synchronized ConnectorRegistry getInstance() {
        if (connectorRegistryInstance == null) {
            connectorRegistryInstance = new ConnectorRegistry();
            _logger.log(Level.FINE,"creating new connector registry");
        }
        logFine("returning the connector registry");
        return connectorRegistryInstance;
    }
   
    /**
     * Protected constructor.
     * It is protected as it follows singleton pattern.
     */

    protected ConnectorRegistry() {
        resourceAdapters = new HashMap();
        factories = new HashMap();
        resourceAdapterConfig = new HashMap();
        _logger.log(Level.FINE,"initialized the connector registry");
    }

    /** Adds the object implementing ActiveResourceAdapter
     *  interface to the registry.
     *  @param rarModuleName RarName which is the key
     *  @param rar           ActiveResourceAdapter instance which is the value.
     */

    public void addActiveResourceAdapter( String rarModuleName, 
                 ActiveResourceAdapter rar) {
	synchronized( resourceAdapters ) {
            resourceAdapters.put(rarModuleName,rar);
	}
        _logger.log(Level.FINE,
                    "Added the active resource adapter to connector registry", 
                    rarModuleName);
    }

    /** Removes the object implementing ActiveResourceAdapter
     *  interface from the registry.
     *  This method is called whenever an active connector module
     *  is removed from the Connector runtime. [eg. undeploy/recreate etc] 
     *  @param rarModuleName RarName which is the key
     *  @return true if successfully removed
     *          false if deletion fails.
     */

    public boolean removeActiveResourceAdapter(String rarModuleName) {
       	Object o = null;
	
	synchronized( resourceAdapters ) {
	    o = resourceAdapters.remove( rarModuleName ); 
	}
	
        if( o == null) {
            logFine("Failed to remove the resource adapter from connector registry" +
               rarModuleName);
            return false;
        } else {
            _logger.log(Level.FINE,
                 "removed the active resource adapter from connector registry",
                 rarModuleName);
            return true;
        }
    }

    /** Retrieves the object implementing ActiveResourceAdapter interface
     *  from the registry. Key is the rarName.
     *  @param rarModuleName Rar name. It is the key
     *  @return object implementing ActiveResourceAdapter interface
     */

    public ActiveResourceAdapter getActiveResourceAdapter(
                     String rarModuleName) {
        if(rarModuleName != null) {
            _logger.log(Level.FINE,
               "returning/found the resource adapter from connector registry",
               rarModuleName);
            return (ActiveResourceAdapter) resourceAdapters.get(rarModuleName);
        } else {
            _logger.log(Level.FINE,
               "Resourceadapter not found in connector registry.Returning null",
               rarModuleName);
            return null;
        }
    }

    /** Checks if the MCF pertaining to the pool is instantiated and present
     * in the registry. Each pool has its own MCF instance.
     * @param poolName Name of the pool
     * @return true if the MCF is found.
     *         false if MCF is not found
     */

    public boolean isMCFCreated (String poolName) {
        boolean created = factories.containsKey( poolName );
        logFine("isMCFCreated " + poolName + " - " + created);
        return created;
    }

    /** Remove MCF instance pertaining to the poolName from the registry.
     *  @param poolName Name of the pool
     *  @return true if successfully removed.
     *          false if removal fails.
     */

    public boolean removeManagedConnectionFactory(String poolName) {
        if(factories.remove(poolName) == null) {
            _logger.log(Level.FINE,
               "Failed to remove the MCF from connector registry.", poolName);
            return false;
        } else {
            logFine("removeMCF " + poolName + " - " + !factories.containsKey(poolName));
            return true;
        }
    }

    /** Add MCF instance pertaining to the poolName to the registry.
     *  @param poolName Name of the pool
     *  @param factory MCF instance to be added.
     */

    public void addManagedConnectionFactory(String poolName, 
                                    PoolMetaData pmd ) {
        factories.put(poolName, pmd);
        logFine("Added MCF to connector registry for: "+  poolName);
    }

    /** Retrieve MCF instance pertaining to the poolName from the registry.
     *  @param poolName Name of the pool
     *  @return factory MCF instance retrieved.
     */
    
    public ManagedConnectionFactory getManagedConnectionFactory(
                     String poolName) {
        if(poolName != null) {
            _logger.log(Level.FINE,
                 "Returning the MCF from connector registry.", poolName);
            
            PoolMetaData pmd = (PoolMetaData) factories.get( poolName );
	    if (pmd != null) {
                return pmd.getMCF();
            }

        } 
        return null;
    }
   

    /** Checks whether the rar is already deployed i.e registered with
     *  connector registry
     *  @param rarModuleName rar Name.
     *  @return true if rar is registered
     *          false if rar is not registered.
     */

    public boolean isRegistered(String rarModuleName) {
        _logger.log(Level.FINE,
            "Checking for MCF presence in connector registry.",rarModuleName);
        synchronized(resourceAdapters ) { 
            return resourceAdapters.containsKey(rarModuleName);
	}
    }

    /** Gets the connector descriptor pertaining the rar
     *  @param rarModuleName rarName
     *  @return ConnectorDescriptor which represents the ra.xml
     */

    public ConnectorDescriptor getDescriptor(String rarModuleName) {
        ActiveResourceAdapter ar = null;
        if(rarModuleName != null) {
            ar = (ActiveResourceAdapter) resourceAdapters.get(rarModuleName);
        }
        if (ar != null) {
            _logger.log(Level.FINE,
                 "Found/returing Connector descriptor in connector registry.", 
                 rarModuleName);
            return ar.getDescriptor();
        } else { 
            _logger.log(Level.FINE,
                 "Couldnot find Connector descriptor in connector registry.", 
                 rarModuleName);
            return null;
        }
    }

    /** Gets the runtime equivalent of policies enforced by the Security Maps 
     *  pertaining to a pool from the Pool's Meta Data.   
     *  @param poolName Name of the pool
     *  @return runtimeSecurityMap in the form of HashMap of
     *   HashMaps (user and groups). 
     *  @see SecurityMapUtils.processSecurityMaps(SecurityMap[])
     */

    public RuntimeSecurityMap getRuntimeSecurityMap(String poolName) {
        if(poolName != null) {
            _logger.log(Level.FINE,
              "Returing the security map from connector registry.", poolName);
            PoolMetaData pmd = (PoolMetaData)factories.get(poolName);
            return pmd.getRuntimeSecurityMap();
        } else {
            return null;
        }
    }

    /** Get the resource adapter config properties object registered with 
     *  registry against the rarName.
     *  @param rarName Name of the rar
     *  @return ResourceAdapter configuration object
     */

    public ResourceAdapterConfig getResourceAdapterConfig(String rarName) {
        if(rarName != null) {
            _logger.log(Level.FINE,
              "Returing the resourceadapter Config from registry.",rarName);
            return (ResourceAdapterConfig)resourceAdapterConfig.get(rarName);
        } else {
            return null;
        }
    }

    /** Add the resource adapter config properties object to registry 
     *  against the rarName.
     *  @param rarName Name of the rar
     *  @param raConfig ResourceAdapter configuration object
     */

    public void addResourceAdapterConfig(String rarName,
                 ResourceAdapterConfig raConfig) {
        if( rarName != null) {
            _logger.log(Level.FINE,
                 "Adding the resourceAdapter Config to connector registry.", 
                 rarName);
            resourceAdapterConfig.put(rarName,raConfig);
        }
    }

    /** Remove the resource adapter config properties object from registry 
     *  @param rarName Name of the rar
     *  @return true if successfully deleted
     *          false if deletion fails
     */

    public boolean removeResourceAdapterConfig(String rarName) {
        if(resourceAdapterConfig.remove(rarName) == null) {
            _logger.log(Level.FINE,
                 "failed to remove the resourceAdapter config from registry.", 
                 rarName);
            return false;
        } else {
            _logger.log(Level.FINE,
                 "Removed the resourceAdapter config map from registry.", 
                 rarName);
            return true;
        }
    }

    /**
     * Returns all Active Resource Adapters in the connector runtime.
     * @return All active resource adapters in the connector runtime
     */
    public ActiveResourceAdapter[] getAllActiveResourceAdapters() {
        return (ActiveResourceAdapter[])
                this.resourceAdapters.values().
                        toArray(new ActiveResourceAdapter[]{});
    }


    static private void logFine( String msg ) {
        if ( msg != null && _logger.isLoggable(Level.FINE) ) {
	    _logger.fine( msg );
	}
    }

    public PoolMetaData getPoolMetaData( String poolName ) {
        return (PoolMetaData) factories.get( poolName );
    }
}
