/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.connectors;

import com.sun.enterprise.connectors.util.ResourcesUtil;
import java.util.Hashtable;
import java.util.logging.*;
import javax.naming.*;

import com.sun.enterprise.*;
import com.sun.enterprise.deployment.*;
import com.sun.enterprise.server.*;

/**
 * This is connector resource admin service. It creates and deletes the
 * connector resources.
 * @author    Srikanth P 
 */


public class ConnectorResourceAdminServiceImpl extends 
                     ConnectorServiceImpl implements ConnectorAdminService {
     
    /**
     * Default constructor
     */

     public ConnectorResourceAdminServiceImpl() {
         super();
     }

    /** 
     * Creates the connector resource on a given connection pool
     * @param jndiName JNDI name of the resource to be created
     * @poolName PoolName to which the connector resource belongs.
     * @resourceType Resource type Unused.
     * @throws ConnectorRuntimeException If the resouce creation fails.
     */

    public void createConnectorResource(String jndiName, String poolName, 
                    String resourceType) throws ConnectorRuntimeException
    {

        String errMsg = "rardeployment.jndi_lookup_failed";
        String name = poolName;
        try {
	    ConnectorConnectionPool connectorConnectionPool = null;
            String jndiNameForPool = ConnectorAdminServiceUtils.
                getReservePrefixedJNDINameForPool(poolName);
            InitialContext ic = new InitialContext();
            try {
                connectorConnectionPool = 
                    (ConnectorConnectionPool) ic.lookup(jndiNameForPool);
            } catch(NamingException ne) {
                checkAndLoadPoolResource(poolName);
            }

            connectorConnectionPool = 
                    (ConnectorConnectionPool) ic.lookup(jndiNameForPool);
            ConnectorDescriptorInfo cdi = connectorConnectionPool.
                getConnectorDescriptorInfo();

            javax.naming.Reference ref=new  javax.naming.Reference( 
                   connectorConnectionPool.getConnectorDescriptorInfo().
                   getConnectionFactoryClass(), 
                   "com.sun.enterprise.naming.factory.ConnectorObjectFactory",
                   null);
            StringRefAddr addr = new StringRefAddr("poolName",poolName);
            ref.add(addr);
            addr = new StringRefAddr("rarName", cdi.getRarName() );
            ref.add(addr);

            errMsg = "Failed to bind connector resource in JNDI";
            name = jndiName;
            Switch.getSwitch().getNamingManager().publishObject(
                          jndiName,ref,true);
            //To notify that a connector resource rebind has happened.
            ConnectorResourceNamingEventNotifier.getInstance().
                    notifyListeners(
                            new ConnectorNamingEvent(
                                    jndiName,ConnectorNamingEvent.EVENT_OBJECT_REBIND));

        } catch(NamingException ne) {
            ConnectorRuntimeException cre = 
                  new ConnectorRuntimeException(errMsg);
            cre.initCause(ne);
            _logger.log(Level.SEVERE,errMsg, name); 
            _logger.log(Level.SEVERE,"", cre); 
            throw cre;
        }
    } 

    /**
     * Deletes the connector resource.
     * @param jndiName JNDI name of the resource to delete.
     * @throws ConnectorRuntimeException if connector resource deletion fails.
     */

    public void deleteConnectorResource(String jndiName) 
                       throws ConnectorRuntimeException 
    {

        try {
            InitialContext ic = new InitialContext();
            ic.unbind(jndiName);
        } catch(NamingException ne) {
            ResourcesUtil resUtil = ResourcesUtil.createInstance();
            if(resUtil.resourceBelongsToSystemRar(jndiName)) {
                return;
            }
            if(ne instanceof  NameNotFoundException){
                _logger.log(Level.FINE,
                    "rardeployment.connectorresource_removal_from_jndi_error",
                    jndiName);
                _logger.log(Level.FINE,"", ne);
                return;
            }
            ConnectorRuntimeException cre =  new ConnectorRuntimeException(
                            "Failed to delete connector resource from jndi");
            cre.initCause(ne);
            _logger.log(Level.SEVERE,
                    "rardeployment.connectorresource_removal_from_jndi_error",
                    jndiName);
            _logger.log(Level.SEVERE,"", cre);
            throw cre;
        }
    } 

    /**
     * If the suffix is one of the valid context return true. 
     * Return false, if that is not the case.
     */
    public boolean isValidJndiSuffix(String suffix) {
        if (suffix != null) {
            for (String validSuffix : ConnectorConstants.JNDI_SUFFIX_VALUES) {
                if (validSuffix.equals(suffix)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Look up the JNDI name with appropriate suffix.
     * Suffix can be either __pm or __nontx.
     */
    public Object lookup(String name) throws NamingException {
        Hashtable ht = null;
        String suffix = getValidSuffix(name); 
        if (suffix != null) {
            ht = new Hashtable();
            ht.put(ConnectorConstants.JNDI_SUFFIX_PROPERTY, suffix);
            name = name.substring(0, name.lastIndexOf(suffix));
        }
        InitialContext ic = new InitialContext(ht);
        return ic.lookup(name);
    }

    /**
     *  Gets Connector Resource Rebind Event notifier.
     * @return  ConnectorNamingEventNotifier
     */
    public ConnectorNamingEventNotifier getResourceRebindEventNotifier(){
        return ConnectorResourceNamingEventNotifier.getInstance();
    }

    private String getValidSuffix(String name) {
        if (name != null) {
            for (String validSuffix : ConnectorConstants.JNDI_SUFFIX_VALUES) {
                if (name.endsWith(validSuffix)) {
                    return validSuffix;
                }
            }
        }
        return null;
    }
}
