/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.connectors.util;

import com.sun.enterprise.deployment.*;
import com.sun.enterprise.config.serverbeans.ElementProperty;
import com.sun.enterprise.connectors.*;
import com.sun.enterprise.util.*;
import com.sun.logging.LogDomains;
import java.util.logging.*;
import java.util.*;
import java.lang.*;
import java.lang.reflect.*;
import java.io.IOException;
import org.xml.sax.SAXParseException;

/**
 *  This is managed connection factory configuration parser. It parses the 
 *  ra.xml file for the managed connection factory specific configurations 
 *  like managed connection factory javabean  properties .
 *
 *  @author      Srikanth P
 *
 */

public class MCFConfigParserImpl implements MCFConfigParser {

    static Logger _logger = LogDomains.getLogger(LogDomains.RSR_LOGGER);
   
    /**
     *  Default constructor.
     *
     */

    public MCFConfigParserImpl() {

    }

    /* Parses the ra.xml and returns all the connection definition names. 
     * Since there is no specific connection definition attribute in the 
     * <connection-definition element>, connection factory interface is 
     * taken as the connection definition name. 
     *
     * @param desc ConnectorDescriptor pertaining to rar.
     * @return Array of Connection definition names.
     * @throws  ConnectorRuntimeException If moduleDir is null.
     *          If corresponding rar is not deployed. 
     *
     */

    public String[] getConnectionDefinitionNames(ConnectorDescriptor desc)
               throws ConnectorRuntimeException 
    {

        if(desc == null) {
            throw new ConnectorRuntimeException("Invalid arguments");
        }

        ConnectionDefDescriptor cdd[] = ddTransformUtil.getConnectionDefs(desc);

        String[] connDefNames = null;
        if(cdd != null) {
            connDefNames = new String[cdd.length];
            for(int i=0;i<cdd.length;++i) {
                connDefNames[i] = cdd[i].getConnectionFactoryIntf();
            }
        }
        return connDefNames;
    }

    /** Parses the ra.xml for the managed connection factory javabean 
     *  properties. The managed connection factory to be parsed is 
     *  identified by the moduleDir where ra.xml is present and the 
     *  connection definition name .
     *
     *  Connection definition name  will be unique in a given ra.xml.
     *
     *  It throws ConnectorRuntimeException if either or both the
     *  parameters are null, if corresponding rar is not deployed,
     *  if no connection definition name is found in ra.xml. If rar is deployed
     *  and connection definition name is present but no properties are
     *  present for the corresponding connection definition name, 
     *  null is returned.
     *
     *  @param  desc ConnectorDescriptor pertaining to rar.
     *  @param  connectionDefName connection definition name which is unique
     *          across all the <connection-definition> elements in a given rar.
     *  @return Javabean properties with the propety names and values
     *          of propeties. The property values will be the values
     *          mentioned in ra.xml if present. Otherwise it will be the
     *          default values obtained by introspecting the javabean.
     *          In both the case if no value is present, empty String is
     *          returned as the value.
     *  @throws  ConnectorRuntimeException if either of the parameters are null.
     *           If corresponding rar is not deployed i.e moduleDir is invalid. 
     *           If no connection definition name is found in ra.xml
     */

    public Properties getJavaBeanProps(ConnectorDescriptor desc, 
               String connectionDefName, String rarName) throws ConnectorRuntimeException 
    {

        if(desc == null || connectionDefName == null) {
            throw new ConnectorRuntimeException("Invalid arguments");
        }
        OutboundResourceAdapter ora = 
                      desc.getOutboundResourceAdapter();
        if(ora == null || ora.getConnectionDefs().size() == 0) {
            return null;
        }
        Set connectionDefs = ora.getConnectionDefs();
        if(connectionDefs== null || connectionDefs.size() == 0) {
            return null;
        }
        Iterator iter = connectionDefs.iterator();
        ConnectionDefDescriptor cdd = null;
        boolean connectionDefFound=false;
        while(iter.hasNext()) {
            cdd = (ConnectionDefDescriptor)iter.next();
            if(connectionDefName.equals(cdd.getConnectionFactoryIntf())) {
                connectionDefFound=true;
                break;
            }
        }

        if(connectionDefFound == false) {
            _logger.log(Level.FINE,
                    "No such connectiondefinition found in ra.xml", 
                    connectionDefName);
            throw new ConnectorRuntimeException(
                  "No such connectiondefinition found in ra.xml : " + 
                  connectionDefName);
        }

        /* ddVals           -> Properties present in ra.xml
        *  introspectedVals -> All properties with values
        *                      obtained by introspection of resource
        *                      adapter javabean
        *  mergedVals       -> merged props of raConfigPros and
        *                      allraConfigPropsWithDefVals
        */

        Properties mergedVals = null;
        Set ddVals = cdd.getConfigProperties();
        String className = cdd.getManagedConnectionFactoryImpl();
        if(className != null && className.length() != 0) {
            Properties introspectedVals = configParserUtil.introspectJavaBean(
                               className,ddVals, true, rarName);
            mergedVals = configParserUtil.mergeProps(ddVals,introspectedVals);
        }
        return mergedVals;
    }

}
