
/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * AutoDeployedFilesManager.java
 *
 * Created on September 4, 2003
 *
 */

package com.sun.enterprise.deployment.autodeploy;

import java.io.*;
import java.util.*;
import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.enterprise.server.Constants;

/**
 *  Contains the status of list of files that have been autodeployed. 
 *
 *  @author Mahesh Rangamani
*/

public class AutoDeployedFilesManager {
    
    private static final Logger sLogger=AutoDeployControllerImpl.sLogger;
    static final String STATUS_DIR_NAME = ".autodeploystatus";
    protected  String statDir = null;


    public AutoDeployedFilesManager() {
    }
    
    protected AutoDeployedFilesManager(String s) {
        statDir = s;
    }
    
    /**
     * Create an instance from the persisted file in the specified directory.
     * @param statusDir Directory in which the status file is to read.
    */
    public static AutoDeployedFilesManager loadStatus(File statusDir) throws Exception {
        return loadStatus(statusDir.getAbsolutePath());
    }
    
    public static AutoDeployedFilesManager loadStatus(String autoDeploymentDir) throws Exception {

        String statusDir = autoDeploymentDir + File.separator + STATUS_DIR_NAME;
        String sysAppDirPath = System.getProperty(Constants.INSTALL_ROOT) +
            File.separator + Constants.LIB + File.separator + 
            Constants.LIB_INSTALL + File.separator + 
            Constants.LIB_INSTALL_APPLICATIONS;

        File fileObj = new File(statusDir);
   
        // if the .autodeploystatus directory does not exist 
        // and this is not for auto deploying system applications
        // create the .autodeploystatus directory
        if (!fileObj.exists() && !autoDeploymentDir.equals(sysAppDirPath)) {
        	sLogger.log(Level.INFO, "autoDeployment status dir missing, creating a new one");
        	fileObj.mkdirs();
        }
        AutoDeployedFilesManager adfm = new AutoDeployedFilesManager(statusDir);
        return adfm;
        
    }
    
    public void writeStatus() throws Exception {
           // Nothing to do 
    }
       
    /**
     * Update the status of the file as deployed. 
     *   
     */
    public void setDeployedFileInfo(File f) throws Exception {
      try {
        File statusFile = getStatusFile(f);
        statusFile.createNewFile();
        statusFile.setLastModified(f.lastModified());
        setJbiDeployedFileInfo(f);
      } catch (Exception e) { throw e; }
    }

    /**
     * Create the JBI status file.
     */
    private void setJbiDeployedFileInfo(File f) throws Exception {
      try {
        JBIAutoDeployer jad = JBIAutoDeployer.getInstance();
        File statusFile = jad.getStatusFile(f, new File(statDir));
        if (statusFile != null)
            statusFile.createNewFile();
      } catch (Exception e) { throw e; }
    }
   
    /*
     * Delete status info for file
     */ 
    public void deleteDeployedFileInfo(File f) throws Exception {
      try {
        File statusFile = getStatusFile(f);
        statusFile.delete();
        deleteJbiDeployedFileInfo(f);
      } catch (Exception e) { throw e;}
    }

    /**
     * Delete the JBI status file.
     */
    private void deleteJbiDeployedFileInfo(File f) throws Exception {
      try {
        JBIAutoDeployer jad = JBIAutoDeployer.getInstance();
        File statusFile = jad.getStatusFile(f, new File(statDir));
        if (statusFile != null)
            statusFile.delete();
      } catch (Exception e) { throw e;}
    }
    
    // calculate the status file path. 
    private File getStatusFile(File f) {
        File outDir = new File(this.statDir);
        outDir = obtainFileStatusDir(f, outDir, outDir.getParentFile());
        return new File(outDir, f.getName());
    }

    static File obtainFileStatusDir(File f, File statDir, File autoDeployDir) {
        File dir = f.getParentFile();
        while (!dir.getAbsolutePath().equals(autoDeployDir.getAbsolutePath())) {
            statDir = new File(statDir, dir.getName()); 
            dir = dir.getParentFile();
        }
        statDir.mkdirs();
        return statDir;
    }
   
    /**
      * Compare the list of files with the current status info 
      * and determine the files that need to be deployed 
      * 
      */
    public File[] getFilesForDeployment(File[] latestFiles) {

        if (latestFiles == null) return new File[0];
        File statusDir = new File(statDir);

        ArrayList<File> arrList = new ArrayList<File>();
        for (File deployDirFile : latestFiles) {
            File statusFile = getStatusFile(deployDirFile);
            if (!statusFile.exists() || deployDirFile.lastModified() != statusFile.lastModified()) {            
                arrList.add(deployDirFile);
            }
        }
        return arrList.toArray(new File[0]);
        
    }
    
    /**
      * Compare the list of files with the current status info 
      * and determine the apps that need to be undeployed. 
      * 
      */
    public File[] getFilesForUndeployment(File[] latestFiles) {
        
        File statusDir = new File(statDir);  
        Set<File> statusFiles = 
                AutoDeployDirectoryScanner.getListOfFilesAsSet(statusDir, true);
        
        // The file might have been deleted. In that case, return null.
        if (statusFiles == null || statusFiles.isEmpty()){
            return null;
        }
        
        // now let's get the latestFiles status files names to remove them 
        // from the list of status files, any remaining ones will need to 
        // be undeployed
        for (File deployDirFile : latestFiles) {
            statusFiles.remove(getStatusFile(deployDirFile));
        }
        ArrayList<File> appNames = new ArrayList<File>();
        File autodeployDir = statusDir.getParentFile();        
        for(File statusDirFile : statusFiles) {
            
            // calculate the original file as it was copied in the autodeploy
            // directory
            File filePath = statusDir.getParentFile();
            File f = statusDirFile.getParentFile();
            while (!f.equals(statusDir)) {
                filePath = new File(filePath, f.getName());
                f = f.getParentFile();
            }
            filePath  = new File(filePath,  statusDirFile.getName());
            appNames.add(filePath);
        }
        return appNames.toArray(new File[0]);
    }    
}


