/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.diagnostics;
import com.sun.logging.LogDomains;

import java.util.logging.Logger;
import java.util.logging.Level;
import java.util.List;
import java.util.ArrayList;
import java.io.File;
import java.util.Iterator;
/**
 *
 * @author mu125243
 */
public abstract class TargetResolver {
    
    protected String target;
    protected String repositoryName;
    protected String repositoryDir;
    protected boolean local;
    protected TargetType type;
    protected List<String> instances = new ArrayList(1);
    protected ReportTarget reportTarget;
    protected List<ServiceConfig> serviceConfigs = new ArrayList(1);
    protected ExecutionContext context;
    
    protected static final String DEFAULT_FEATURES_PROPERTY_CLASS =
            "com.sun.enterprise.admin.pluggable.PEClientPluggableFeatureImpl";
    
    protected Logger logger =
            LogDomains.getLogger(LogDomains.ADMIN_LOGGER);
    /** Creates a new instance of TargetResolver */
    public TargetResolver(String target, String repositoryDir, boolean local) {
        this.target = target;
        this.repositoryDir = repositoryDir;
        this.local = local;
    }
    
    public ReportTarget resolve() throws DiagnosticException {
        if(validateTarget()) {
            createTarget();
            return reportTarget;
        }
        throw new InvalidTargetException(" Unable to resolve target : " + target);
    }
    
    
    public abstract boolean validateTarget() throws DiagnosticException;

    
    public ExecutionContext getExecutionContext() {
        return context;
    }
    
    
     public List<ServiceConfig> getServiceConfigs() {
        return serviceConfigs;
     }
     
 
   
    protected void createTarget() throws DiagnosticException {
        setExecutionContext();
        initLogger();
        determineTargetDetails();
        createTargetObject();
    }
    
    protected abstract void setExecutionContext();
    
    protected void determineTargetDetails() {
        //determineTargetDir();
        determineTargetType();
        determineRepositoryDetails();
        determineInstances();
        determineServiceConfigs();
        logger.log(Level.FINE,"diagnostic-service.target_details",
                new Object[] {target, repositoryDir, type, instances});
    }
    
    protected  abstract void determineRepositoryDetails() ;
    protected abstract void determineTargetType() ;
    protected abstract void determineInstances();
    
    protected void setTargetType(TargetType type) {
        this.type = type;
    }

    protected void addInstance(String instanceName) {
        instances.add(instanceName);
    }
    
    protected void determineServiceConfigs() {
        if(instances != null) {
            Iterator<String> iterator = instances.iterator();
            while(iterator.hasNext()) {
                addServiceConfig(iterator.next());
            }
        }
    }
    
    protected void  addServiceConfig(String instanceName) {
        String instanceRepDir = repositoryDir + File.separator +
                            repositoryName ;

        if ((!(instanceName.equals(Constants.SERVER))) &&
                (!(type.equals(TargetType.INSTANCE)))){
            // for example in case of node agent this value would be
            //nodeagents/na1/instance1
            instanceRepDir = instanceRepDir + File.separator + instanceName;
        }
                    
                
        ServiceConfig instanceConfig = ServiceConfigFactory.getInstance().
                        getServiceConfig(context.isLocal(), instanceRepDir, 
                        instanceName);
        if(instanceConfig != null)
            serviceConfigs.add(instanceConfig);
    }
    
    
    protected void createTargetObject() {
        reportTarget = new ReportTarget(repositoryDir, repositoryName, target,
        type, instances, local);
    }
    
    private void initLogger() {
        logger = context.getLogger();
    }

    private Logger getLogger() {
        return logger;
    }
 
}
