/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.resource;

import javax.transaction.*;

/**
 * This resource pool is created when connection pooling is switched off
 * Hence no pooling happens in this resource pool
 *
 * @author Kshitiz Saxena
 * @since 9.1
 */
public class UnpooledResource extends AbstractResourcePool{
   
    private int poolSize;
    
    /** Creates a new instance of UnpooledResourcePool */
    public UnpooledResource(String poolName) throws PoolingException {
        super(poolName);
        
        //No monitoring data for this pool as pooling is switched off
        monitoringEnabled = false;
        //No pool is being maintained, hence no pool cleanup is needed 
        //in case of failure
        failAllConnections = false;
    }

    @Override
    protected synchronized void initPool(ResourceSpec resourceSpec,
            ResourceAllocator allocator)
            throws PoolingException {
        
        if (poolInitialized) {
            return;
        }
        
        //nothing needs to be done as pooling is disabled
        poolSize = 0;
        
        poolInitialized = true;
    }
    
    @Override
    protected ResourceHandle prefetch(ResourceSpec spec, ResourceAllocator alloc, 
            Transaction tran) {
        return null;
    }

    @Override
    protected ResourceHandle getUnenlistedResource(ResourceSpec spec, ResourceAllocator alloc,
            Transaction tran) throws PoolingException {
        ResourceHandle handle = null;
        
        if(incrementPoolSize()){
            try{
                handle = createSingleResource(alloc);
            }catch (PoolingException ex){
                decrementPoolSize();
                throw ex;
            }
            ResourceState state = new ResourceState();
            handle.setResourceState(state);
            state.setEnlisted(false);
            setResourceStateToBusy(handle);
            return handle;
        }
        _logger.info("Fail as poolSize : " + poolSize);
        String msg = localStrings.getStringWithDefault(
                "poolmgr.max.pool.size.reached",
                "In-use connections equal max-pool-size therefore cannot allocate any more connections.");
        throw new PoolingException(msg);
    }
    
    @Override
    public void resourceErrorOccurred(ResourceHandle resourceHandle) throws IllegalStateException {
        freeResource(resourceHandle);
    }
    
    @Override
    protected void freeResource(ResourceHandle resourceHandle){
        decrementPoolSize();
        destroyResource(resourceHandle);
    }
    
    private synchronized boolean incrementPoolSize(){
        if(poolSize >= maxPoolSize)
            return false;
        poolSize++;
        return true;
    }
    
    private synchronized void decrementPoolSize(){
        poolSize--;
    }
}
