/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.webservice;

import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.Map;

import java.security.PrivilegedAction;
import java.security.PrivilegedExceptionAction;
import java.security.PrivilegedActionException;
import javax.security.auth.Subject;
import javax.security.auth.message.config.*;
import javax.security.auth.message.AuthException;
import javax.security.auth.message.AuthStatus;

import com.sun.enterprise.security.jmac.PacketMessageInfo;
import com.sun.enterprise.security.jmac.PacketMapMessageInfo;
import com.sun.enterprise.security.jmac.config.PipeHelper;
import com.sun.enterprise.util.LocalStringManagerImpl;
import com.sun.logging.LogDomains;

import com.sun.xml.ws.api.SOAPVersion;
import com.sun.xml.ws.api.model.wsdl.WSDLPort;
import com.sun.xml.ws.api.pipe.Pipe;
import com.sun.xml.ws.api.pipe.PipeCloner;
import com.sun.xml.ws.api.pipe.helper.AbstractFilterPipeImpl;
import com.sun.xml.ws.api.message.Message;
import com.sun.xml.ws.api.message.Messages;
import com.sun.xml.ws.api.message.Packet;

/**
 * This pipe is used to do 196 security
 */
public class CommonServerSecurityPipe extends AbstractFilterPipeImpl {

    protected static Logger _logger = LogDomains.getLogger(
        LogDomains.SECURITY_LOGGER);
    protected static LocalStringManagerImpl localStrings = 
        new LocalStringManagerImpl(CommonServerSecurityPipe.class);

    private final boolean isHttpBinding;

    private PipeHelper helper;

    CommonServerSecurityPipe(Map props, final Pipe next, 
			     boolean isHttpBinding) {

        super(next);

	props.put(PipeConstants.SECURITY_PIPE,this);

	this.helper = new PipeHelper(PipeConstants.SOAP_LAYER,props,null);

        this.isHttpBinding = isHttpBinding;
       
    }    
    
    protected CommonServerSecurityPipe(CommonServerSecurityPipe that,
            PipeCloner cloner) {

        super(that, cloner);

	// XXX all the cloned pipes will share the same helper
	// is that what we want?
        this.helper = that.helper;

        this.isHttpBinding = that.isHttpBinding;

    }

    public void preDestroy() {
	// XXX all the cloned pipes will share the same helper
	// is it ok to disable the helper
	helper.disable();
    }    
    
    public Pipe copy(PipeCloner cloner) {
        return new CommonServerSecurityPipe(this, cloner);
    }
    
    public Packet process(Packet request) {

        if (isHttpBinding) {
            return next.process(request);
        }
        
        Packet response = null;

        try {

            response = processRequest(request);

        } catch(Exception e) {
            if (_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.FINE, "Failure in security pipe process", e);
            }
	    if (response == null) {
		response = new Packet();
	    }
            response = response.createResponse
		(Messages.create(e, SOAPVersion.SOAP_11));
        }

        return response;
    }    

    private Object wsdlModelName() { 
	WSDLPort wsdlModel =
	    (WSDLPort) helper.getProperty(PipeConstants.WSDL_MODEL);
	return (wsdlModel == null ? "unknown" : wsdlModel.getName());
    }

    private Packet processRequest(Packet request) throws Exception {

        AuthStatus status = AuthStatus.SUCCESS;
	
	PacketMessageInfo info= new PacketMapMessageInfo(request,new Packet());

	// XXX at this time, we expect the server subject to be null

	Subject serverSubject = (Subject) 
	    request.invocationProperties.get(PipeConstants.SERVER_SUBJECT);

	//could change the request packet
        ServerAuthContext sAC = 
	    helper.getServerAuthContext(info,serverSubject);

	Subject clientSubject = getClientSubject(request);

	final Packet validatedRequest;
	
	try {

	    if (sAC != null) {
		
		// client subject must not be null
		// and when return status is SUCCESS, module
		// must have called handler.handle(CallerPrincipalCallback)

		status = sAC.validateRequest(info,clientSubject,serverSubject);

	    }
	} catch(Exception e) {

	    _logger.log(Level.SEVERE,"ws.error_validate_request", e);

	    throw new IllegalStateException
		(localStrings.getLocalString
		 ("enterprise.webservice.cantValidateRequest",
		  "Cannot validate request for {0}",
		  new Object[] { wsdlModelName() }),e);

	} finally {

	    validatedRequest = info.getRequestPacket();
	    helper.auditInvocation(validatedRequest,status);
	}

	Packet response;

	if (status == AuthStatus.SUCCESS) {

	    helper.authorize(validatedRequest);

	    // proceed to invoke the endpoint
	    // only do doAsPriv if SecurityManager is in effect

	    if (System.getSecurityManager() == null) {
		response = next.process(validatedRequest);
	    } else {
		try {
		    response = (Packet)Subject.doAsPrivileged
			(clientSubject,new PrivilegedExceptionAction() {
			    public Object run() throws Exception {
				return next.process(validatedRequest);
			    }
			}, null);
		} catch (PrivilegedActionException pae) {
		    Throwable cause = pae.getCause();
		    if (cause instanceof AuthException){
			_logger.log(Level.SEVERE,"ws.error_next_pipe", cause);
		    }
		    if (cause instanceof Exception) {
			throw (Exception)cause;
		    } else {
			throw new Exception(cause);
		    }
		}
	    }

	    if (sAC != null) {

		// check for SOAP fault
		Message m = response.getMessage();
		if (m == null || m.isFault()) {
		    return response;
		}

		info.setResponsePacket(response);

		response = processResponse(info, sAC, serverSubject);
	    }

	} else {
	    if (_logger.isLoggable(Level.FINE)) {
		_logger.log(Level.FINE,"ws.status_validate_request", status);
	    }
	    response = info.getResponsePacket();
	}

        return response;
    }

    private Packet processResponse(PacketMessageInfo info,
				   ServerAuthContext sAC,
				   Subject serverSubject) throws Exception {
        
        AuthStatus status = AuthStatus.SUCCESS;

	try {
	    status = sAC.secureResponse(info, serverSubject);

	} catch (Exception e) {
	    if (e instanceof AuthException) {
		if (_logger.isLoggable(Level.INFO)) {
		    _logger.log(Level.INFO, "ws.error_secure_response", e);
		}
	    } else {
		_logger.log(Level.SEVERE, "ws.error_secure_response", e);
	    }
    
	    throw e;
	}

	return info.getResponsePacket();

    }

    private static Subject getClientSubject(Packet p) {
	
	Subject s = null;
	
	if (p != null) {

	    s =(Subject) 
		p.invocationProperties.get(PipeConstants.CLIENT_SUBJECT);

	}
	if (s == null) {
	    
	    s = PipeHelper.getClientSubject();
	    
	    if (p != null) {
		p.invocationProperties.put(PipeConstants.CLIENT_SUBJECT,s);
	    }
	}

	return s;
    }

}







