/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.grizzly.rails;

import com.sun.enterprise.web.connector.grizzly.DefaultProcessorTask;
import com.sun.enterprise.web.connector.grizzly.ProcessorTask;
import java.io.IOException;
import java.util.logging.Level;

import com.sun.enterprise.web.connector.grizzly.SelectorThread;
import org.apache.coyote.Adapter;

/**
 * JRuby on rails implementation of Grizzly SelectorThread
 * @author TAKAI Naoto
 * @author Jeanfrancois Arcand
 */
public class RailsSelectorThread extends SelectorThread {
    /**
     * Is Grizzly embedded in GlassFish.
     */
    protected static boolean embeddedInGlassFish = false;

    // Are we running embedded or not.
    static {
        try {
            embeddedInGlassFish = (Class.forName("org.apache.coyote.tomcat5.Constants") != null);
        } catch (Exception ex) {
            ; // Swallow
        }
    }

    private String jrubyLib = null;

    private int numberOfRuntime = 5;

    private RubyObjectPool pool = null;

    private String railsRoot = null;

    public String getRailsRoot() {
        return railsRoot;
    }

    @Override
    public void initEndpoint() throws IOException, InstantiationException {
        if (embeddedInGlassFish) {
            railsRoot = System.getProperty("com.sun.aas.instanceRoot") + "/applications/rails";
        }
        setupSystemProperties();
        initializeRubyRuntime();

        this.adapter = new RailsAdapter(pool);
        
        setBufferResponse(false);

        super.initEndpoint();
    }

    public void setNumberOfRuntime(int numberOfRuntime) {
        this.numberOfRuntime = numberOfRuntime;
    }

    public void setRailsRoot(String railsRoot) {
        this.railsRoot = railsRoot;
    }

    @Override
    public synchronized void stopEndpoint() {
        pool.stop();

        super.stopEndpoint();
    }

    protected void initializeRubyRuntime() {
        pool = new RubyObjectPool();
        pool.setNumberOfRuntime(numberOfRuntime);
        pool.setJrubyLib(jrubyLib);
        pool.setRailsRoot(railsRoot);

        try {
            pool.start();
        } catch (Throwable t) {
            logger.log(Level.WARNING, t.getMessage());
        }
    }

    protected void setupSystemProperties() {
        String jrubyBase = System.getProperty("jruby.base");
        String jrubyHome = System.getProperty("jruby.home");
        String jrubyShell = System.getProperty("jruby.shell");
        String jrubyScript = System.getProperty("jruby.script");

        if (jrubyBase == null) {
            throw new IllegalStateException("Set system property jruby.base to JRuby directory");
        }
        if (jrubyHome == null) {
            System.setProperty("jruby.home", jrubyBase);
        }
        if (jrubyShell == null) {
            System.setProperty("jruby.shell", "");
        }
        if (jrubyScript == null) {
            System.setProperty("jruby.script", "");
        }

        jrubyLib = System.getProperty("jruby.lib");
        if (jrubyLib == null) {
            jrubyLib = jrubyBase + "/lib";
            System.setProperty("jruby.lib", jrubyLib);
        }
    }
    
    
    /**
     * Create <code>ProcessorTask</code> objects and configure it to be ready
     * to proceed request.
     */
    protected ProcessorTask newProcessorTask(boolean initialize){     
            
        // Create a specialized DefaultProcessorTask that doesn't flush
        // the headers since Rails will take care of it.
        DefaultProcessorTask task = 
                new DefaultProcessorTask(initialize, bufferResponse){
                     /**
                     * Prepare and post the response.
                     * @param input the InputStream to read bytes
                     * @param output the OutputStream to write bytes
                     */       
                    public void postResponse() throws Exception{

                        try {
                            adapter.afterService(request,response);
                        } catch (Exception ex) {
                            error = true;
                            SelectorThread.logger().log(Level.FINEST,
                                    "processorTask.errorFinishingRequest", ex);            
                        }

                        // Finish the handling of the request
                        try {
                             inputBuffer.endRequest();
                        } catch (IOException e) {
                            error = true;
                        } catch (Throwable t) {
                            SelectorThread.logger().log(Level.SEVERE,
                                    "processorTask.errorFinishingRequest", t);
                            // 500 - Internal Server Error
                            response.setStatus(500);
                            error = true;
                        }
                        // Force closing the connection. This connection is also
                        // forced to be closed when doing task.setDropConnection(true) below,
                        // except the header value is connection: keep-alive. Setting 
                        // this value will set connection:close. Those headers aren't
                        // flushed but needed by Grizzly to properly manage the connection.
                        keepAlive = false;

                        // If there was an error, make sure the request is counted as
                        // and error, and update the statistics counter
                        if (error) {
                            response.setStatus(500);
                        }

                        if (isMonitoringEnabled()) {
                            request.updateCounters();
                            getAdapter().fireAdapterEvent(
                                    Adapter.REQUEST_PROCESSING_COMPLETED, 
                                    request.getRequestProcessor());              
                        }

                        // Next request
                        inputBuffer.nextRequest();
                        outputBuffer.nextRequest();
                    }                       
            };
        task.setAdapter(adapter);
        task.setMaxHttpHeaderSize(maxHttpHeaderSize);
        task.setBufferSize(requestBufferSize);
        task.setSelectorThread(this);               
        task.setRecycle(recycleTasks);
        task.setDefaultResponseType(defaultResponseType);
        task.setForcedResponseType(forcedResponseType);
        task.setDropConnection(true);
        
        // Asynch extentions
        if ( asyncExecution ) {
            task.setEnableAsyncExecution(asyncExecution);
            task.setAsyncHandler(asyncHandler);          
        }
                
        task.setPipeline(processorPipeline);         
        configureCompression(task);
        
        return (ProcessorTask)task;        
    }
}
