/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.ara;


import com.sun.enterprise.web.connector.grizzly.LinkedListPipeline;
import com.sun.enterprise.web.connector.grizzly.Pipeline;
import com.sun.enterprise.web.connector.grizzly.StreamAlgorithm;
import com.sun.enterprise.web.connector.grizzly.Task;
import com.sun.enterprise.web.connector.grizzly.TaskEvent;
import com.sun.enterprise.web.connector.grizzly.TaskListener;
import com.sun.enterprise.web.ara.algorithms.ContextRootAlgorithm;
import com.sun.enterprise.web.ara.rules.ThreadRatioRule;

import java.util.ArrayList;

import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.List;


/**
 * Customized <code>Pipeline</code> which wrap the original <code>Task</code>
 * instance with an instance of <code>IsolatedTask</code>
 *
 * @author Jeanfrancois Arcand
 */
public class IsolationPipeline extends LinkedListPipeline 
        implements TaskListener{
    
    private final static String ALGORITHM_CLASS = 
         "com.sun.enterprise.web.ara.isolationPipeline.algorithm";
    private final static String RULE_EXECUTOR_CLASS = 
         "com.sun.enterprise.web.ara.isolationPipeline.ruleExecutor";
            
            
    /**
     * Cache instance of <code>IsolatedTask</code>
     */
    private ConcurrentLinkedQueue<IsolatedTask> isolatedTasks;
    
      
    // ------------------------------------------------------ Constructor ---/
    
    
    public IsolationPipeline(){
    }
    
    
    /**
     * Initialize this pipeline by first initializing its parent, and then by
     * creating the caches and the rule executor engine.
     */
    public void initPipeline(){
        // 1. first, init this pipeline.
        super.initPipeline();
        
        // 2. Create cache
        isolatedTasks = new ConcurrentLinkedQueue<IsolatedTask>();
        
        // 3. Cache IsolatedTask
        for (int i=0; i < maxThreads; i++){
            isolatedTasks.offer(newIsolatedTask());
        }  
    }
    
    
    /**
     * Execute the wrapped <code>Task</code>
     */
    public void addTask(Task task) {    
        // SSL not yet supported.
        if (task.getType() == Task.READ_TASK){
            super.addTask(wrap(task));
        } else {
            super.addTask(task);
        }
    }
    

    /**
     * Wrap the current <code>Task</code> using an <code>IsolatedTask</code>
     */
    private Task wrap(Task task){
        IsolatedTask isolatedTask = isolatedTasks.poll();
        if ( isolatedTask == null){
            isolatedTask = newIsolatedTask();
        }
        isolatedTask.wrap(task);
        return isolatedTask;
    }

    
    /**
     * Create a new <code>IsolatedTask</code>
     */
    private IsolatedTask newIsolatedTask(){
        IsolatedTask task = new IsolatedTask();
        
        task.setAlgorithm(newAlgorithm());
        task.setRulesExecutor(newRulesExecutor());
        task.addTaskListener(this);
        task.pipeline = this;
        return task;
    }
    
    
    /**
     * Create a new <code>StreamAlgorithm</code>.
     */
    private StreamAlgorithm newAlgorithm(){
        return (StreamAlgorithm)loadInstance(ALGORITHM_CLASS);
    }
    
    
    /**
     * Create the new <code>RulesExecutor</code>
     */
    private RulesExecutor newRulesExecutor(){
        return (IsolationRulesExecutor)loadInstance(RULE_EXECUTOR_CLASS);
    }
    
       
    // ----------------------------------------------- Task Listener ---------//
    
    public void taskStarted(TaskEvent event) {
        ; // Do nothing.
    }

    
    /**
     * Return the <code>IsolatedTask</code> to the pool.
     */
    public void taskEvent(TaskEvent event) {
        if ( event.getStatus() == TaskEvent.COMPLETED)
            isolatedTasks.offer((IsolatedTask)event.attachement());
    }
    
    // ----------------------------------------------- Util ------------------//
    
    /**
     * Instanciate a class based on a property.
     */
    private Object loadInstance(String property){        
        Class className = null;                               
        Pipeline pipeline = null;                               
        try{                              
            className = Class.forName(property);
            return className.newInstance();
        } catch (ClassNotFoundException ex){
        } catch (InstantiationException ex){
        } catch (IllegalAccessException ex){
        }
        
        // Default
        if ( property.equals(ALGORITHM_CLASS)){
            return new ContextRootAlgorithm();
        } else if ( property.equals(RULE_EXECUTOR_CLASS)){
            return new IsolationRulesExecutor();
        }
        throw new IllegalStateException();
    }
    
}
