/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.web.connector.grizzly;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import org.apache.coyote.OutputBuffer;

import org.apache.coyote.Response;
import org.apache.coyote.http11.InternalOutputBuffer;
import org.apache.tomcat.util.buf.ByteChunk;



/**
 * Output buffer.
 * Buffer the bytes until the <code>ByteChunk</code> is full or the request
 * is completed.
 * 
 * @author Jean-Francois Arcand
 * @author Scott Oaks
 */
public class SocketChannelOutputBuffer extends InternalOutputBuffer{

    /**
     * Underlying output socketChannel.
     */
    protected SocketChannel socketChannel;
    
    
    /**
     * Underlying ByteByteBuffer
     */
    private ByteBuffer outputByteBuffer;
    
    
    /**
     * ACK static bytes.
     */
    private final static ByteBuffer ACK = 
            ByteBuffer.wrap("HTTP/1.1 100 Continue\r\n\r\n".getBytes());
    
    // ----------------------------------------------------------- Constructors
    

    /**
     * Alternate constructor.
     */
    public SocketChannelOutputBuffer(Response response, 
            int headerBufferSize, boolean useSocketBuffer) {
        super(response,headerBufferSize, useSocketBuffer); 
        
        if ( !useSocketBuffer ) {
            outputStream = new NIOOutputStream();
        }
        
        outputByteBuffer = ByteBuffer.allocate(headerBufferSize);
    }

    
    // ------------------------------------------------------------- Properties


    /**
     * Set the underlying socket output stream.
     */
    public void setChannel(SocketChannel socketChannel) {
        this.socketChannel = socketChannel;
    }

    
    /**
     * Return the underlying SocketChannel
     */
    public SocketChannel getChannel(){
        return socketChannel;
    }
    
    // --------------------------------------------------------- Public Methods

    /**
     * Send an acknoledgement without buffering.
     */
    public void sendAck() throws IOException {
        if (!committed)
            flushChannel(ACK.slice());
    }
    
    
    /**
     * Callback to write data from the buffer.
     */
    public void realWriteBytes(byte cbuf[], int off, int len)
        throws IOException {
        try{
            if (len > 0) {
                if ( cbuf.length > outputByteBuffer.capacity()){
                    outputByteBuffer = ByteBuffer.allocate(cbuf.length);
                }
                outputByteBuffer.put(cbuf,off,len);
                outputByteBuffer.flip();
                flushChannel(outputByteBuffer);
            }
        } finally {
            outputByteBuffer.clear();
        }
    }
    
    
    /**
     * Flush the buffer by looping until the <code>ByteBuffer</code> is empty
     * @param bb the ByteBuffer to write.
     */   
    public void flushChannel(ByteBuffer bb) throws IOException{
        OutputWriter.flushChannel(socketChannel, bb);
    }

    
    /**
     * Recycle the output buffer. This should be called when closing the 
     * connection.
     */
    public void recycle() {
        response.recycle();
        socketBuffer.recycle();
        pos = 0;
        lastActiveFilter = -1;
        committed = false;
        finished = false;
        outputByteBuffer.clear();

        socketChannel = null;
    }
    
    
    // ---------------------------------------------- Class helper ----------//
    
    /**
     * OutputBuffer delegating all writes to the <code>OutputWriter</code>
     */
    private final class NIOOutputStream extends OutputStream{        
        public void write(byte[] b, int off, int len) throws IOException{           
            realWriteBytes(b,off,len);
        }
        
        public void write(int b) throws IOException {
            write((byte)b);
        }
        
        public void write(byte b) throws IOException {
            flushChannel(ByteBuffer.wrap(new byte[]{b}));
        }
    }
    
}
