/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.web.connector.grizzly.ssl;

import com.sun.enterprise.web.connector.grizzly.OutputWriter;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLEngineResult.HandshakeStatus;

/**
 * SSL over NIO utility to encrypt <code>ByteBuffer</code> and flush them.
 * All the SSLEngine operations are delegated to class <code>SSLUtils</code>
 *
 * @author Jeanfrancois Arcand
 */
public final class SSLOutputWriter{

    /**
     * Encrypt the response and flush it using <code>OutputWriter</code>
     */     
    public static void flushChannel(SocketChannel socketChannel, ByteBuffer bb)
            throws IOException{
      
        SSLWorkerThread workerThread = (SSLWorkerThread)Thread.currentThread();
        SSLEngine sslEngine = workerThread.getSSLEngine();
        ByteBuffer outputBB = workerThread.getOutputBB();
        flushChannel(socketChannel,bb,outputBB,sslEngine);
    }
    
    
    /**
     * Encrypt the response and flush it using <code>OutputWriter</code>
     */     
    public static void flushChannel(SocketChannel socketChannel, ByteBuffer bb,
            ByteBuffer outputBB, SSLEngine sslEngine) throws IOException{   
        
        while (bb.hasRemaining()) {
            SSLEngineResult result = SSLUtils.wrap(bb,outputBB,sslEngine);
            switch (result.getStatus()) {
                case OK:
                    if (result.getHandshakeStatus() == HandshakeStatus.NEED_TASK) {
                        SSLUtils.executeDelegatedTask(sslEngine);
                    }
                    break;

                default:
                    throw new IOException("SSLOutputWriter: " + result.getStatus());
            }

            if (outputBB.hasRemaining()) {
                OutputWriter.flushChannel(socketChannel,outputBB);
            }
        }
        outputBB.clear();
    }
}
