/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */


package com.sun.jdo.api.persistence.enhancer.classfile;

import java.io.*;

/**
 * ExceptionRange represents a range an exception handler within
 * a method in class file.
 */

public class ExceptionRange {
  /* The start of the exception hander (inclusive) */
  private InsnTarget excStartPC;

  /* The end of the exception hander (exclusive) */
  private InsnTarget excEndPC;

  /* The exception handler code */
  private InsnTarget excHandlerPC;

  /* The exception specification */
  private ConstClass excCatchType;

  /* public accessors */

  /**
   * return the start of the exception hander (inclusive)
   */
  public InsnTarget startPC() {
    return excStartPC;
  }

  /**
   * return the end of the exception hander (exclusive)
   */
  public InsnTarget endPC() {
    return excEndPC;
  }

  /**
   * return the exception handler code
   */
  public InsnTarget handlerPC() {
    return excHandlerPC;
  }

  /** 
   * return the exception specification
   * a null return value means a catch of any (try/finally)
   */
  public ConstClass catchType() {
    return excCatchType;
  }

  /**
   * constructor 
   */

  public ExceptionRange(InsnTarget startPC, InsnTarget endPC,
			InsnTarget handlerPC, ConstClass catchType) {
    excStartPC = startPC;
    excEndPC = endPC;
    excHandlerPC = handlerPC;
    excCatchType = catchType;
  }

  /* package local methods */

  static ExceptionRange read(DataInputStream data, CodeEnv env)
    throws IOException {
    InsnTarget startPC = env.getTarget(data.readUnsignedShort());
    InsnTarget endPC = env.getTarget(data.readUnsignedShort());
    InsnTarget handlerPC = env.getTarget(data.readUnsignedShort());
    ConstClass catchType =
      (ConstClass) env.pool().constantAt(data.readUnsignedShort());
    return new ExceptionRange(startPC, endPC, handlerPC, catchType);
  }

  void write(DataOutputStream out) throws IOException {
    out.writeShort(excStartPC.offset());
    out.writeShort(excEndPC.offset());
    out.writeShort(excHandlerPC.offset());
    out.writeShort(excCatchType == null ? 0 : excCatchType.getIndex());
  }

  void print(PrintStream out, int indent) {
    ClassPrint.spaces(out, indent);
    out.print("Exc Range:");//NOI18N
    if (excCatchType == null)
        out.print("any");//NOI18N
    else
        out.print("'" + excCatchType.asString() + "'");//NOI18N
    out.print(" start = " + Integer.toString(excStartPC.offset()));//NOI18N
    out.print(" end = " + Integer.toString(excEndPC.offset()));//NOI18N
    out.println(" handle = " + Integer.toString(excHandlerPC.offset()));//NOI18N
  }
}

