/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.appserv.ha.spi;

import java.io.Serializable;

/**
 * AttributeMetadata contains the metadata of an attribute that is part of an
 * Http Session. When a container decides to save a session it passes an
 * instance of CompositeMetaData which contains a collection of
 * AttributeMetadata.
 * 
 * <p>
 * The attribute in question could have been deleted, or modified or could be a
 * new attribute inside the HttpSession. getOperation() tells exactly what
 * operation needs to be performed for this attribute
 * 
 * <p>
 * The attribute state/data itself can be obtained with getState(). Since an
 * attribute is part of a session, the attributes must be deleted when the
 * session is removed. The CompositeMetadata contains the last access time and
 * inactive timeout for the session.
 * 
 * @see CompositeMetadata
 */
public final class AttributeMetadata implements Serializable {

    private String attributeName;

    private Operation opcode;

    private byte[] data;

    /**
     * Operation to be performed on this attribute
     */
    public enum Operation {
        ADD, DELETE, UPDATE
    };

    /**
     * Construct an AtributeMetadata
     * 
     * @param attributeName
     *            the attribute name
     * @param opcode
     *            The operation to be performed on the AttrbuteMetadata
     * @param data
     *            The attribute data
     */
    public AttributeMetadata(String attributeName, Operation opcode, byte[] data) {
        this.attributeName = attributeName;
        this.opcode = opcode;
        this.data = data;
    }

    /**
     * Returns name of the attribute
     * 
     * @return attribute name
     */
    public String getAttributeName() {
        return attributeName;
    }

    /**
     * Get the operation to be performed on the attribute.
     * 
     * @return the operation to be performed on this attribute
     */
    public Operation getOperation() {
        return opcode;
    }

    /**
     * Get the attribute data
     * 
     * @return the data
     */
    public byte[] getState() {
        return data;
    }
    
    public boolean equals(Object obj) {
        if(this == obj) {
            return true;
        }
        if(!(obj instanceof AttributeMetadata) || obj == null) {
            return false;
        }
        AttributeMetadata otherAttributeMetadata = (AttributeMetadata)obj;
        if(getAttributeName() == null) {
            if(otherAttributeMetadata.getAttributeName() == null) { 
                return true;
            } else {
                return false;
            }
        } else {
            if(otherAttributeMetadata.getAttributeName() == null) { 
                return false;
            } else {
                return getAttributeName().equalsIgnoreCase(otherAttributeMetadata.getAttributeName());
            }          
        }                
    }
    
    public int hashCode() {
        int hash = 1;
        hash = hash * 31
                + (attributeName == null ? 0 : attributeName.hashCode());
        return hash;
    }
}
