/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.gjc.spi.jdbc40;

import java.sql.SQLException;
import javax.resource.spi.ConnectionManager;
import javax.resource.ResourceException;

import com.sun.gjc.spi.base.DataSource;
import com.sun.gjc.spi.ManagedConnectionFactory;
import com.sun.enterprise.util.i18n.StringManager;


import java.util.logging.Level;

/**
 * Holds the <code>java.sql.Connection</code> object, which is to be
 * passed to the application program.
 *
 * @author Binod P.G
 * @version 1.0, 02/07/31
 */
public class DataSource40 extends DataSource {


    protected final static StringManager localStrings =
            StringManager.getManager(ManagedConnectionFactory.class);

    /**
     * Constructs <code>DataSource</code> object. This is created by the
     * <code>ManagedConnectionFactory</code> object.
     *
     * @param mcf <code>ManagedConnectionFactory</code> object
     *            creating this object.
     * @param cm  <code>ConnectionManager</code> object either associated
     *            with Application server or Resource Adapter.
     */
    public DataSource40(ManagedConnectionFactory mcf, ConnectionManager cm) {
        super(mcf, cm);
    }
    /**
     * Returns an object that implements the given interface to allow access to
     * non-standard methods, or standard methods not exposed by the proxy.
     *
     * If the receiver implements the interface then the result is the receiver
     * or a proxy for the receiver. If the receiver is a wrapper
     * and the wrapped object implements the interface then the result is the
     * wrapped object or a proxy for the wrapped object. Otherwise return the
     * the result of calling <code>unwrap</code> recursively on the wrapped object
     * or a proxy for that result. If the receiver is not a
     * wrapper and does not implement the interface, then an <code>SQLException</code> is thrown.
     *
     * @param iface A Class defining an interface that the result must implement.
     * @return an object that implements the interface. May be a proxy for the actual implementing object.
     * @throws java.sql.SQLException If no object found that implements the interface
     * @since 1.6
     */
    public <T> T unwrap(Class<T> iface) throws SQLException {
        T result ;
        try {
            Object cds = mcf.getDataSource();

            if(iface.isInstance(cds)){
                result = iface.cast(cds);
            }else if (cds instanceof java.sql.Wrapper) {
                result = ((java.sql.Wrapper) cds).unwrap(iface);
            } else {
                String msg = localStrings.getString("jdbc.feature_not_supported");
                throw new SQLException(msg);
            }
        } catch (ResourceException e) {
            _logger.log(Level.WARNING, "jdbc.exc_unwrap", e);
            throw new SQLException(e);
        }
        return result;
    }

    /**
     * Returns true if this either implements the interface argument or is directly or indirectly a wrapper
     * for an object that does. Returns false otherwise. If this implements the interface then return true,
     * else if this is a wrapper then return the result of recursively calling <code>isWrapperFor</code> on the wrapped
     * object. If this does not implement the interface and is not a wrapper, return false.
     * This method should be implemented as a low-cost operation compared to <code>unwrap</code> so that
     * callers can use this method to avoid expensive <code>unwrap</code> calls that may fail. If this method
     * returns true then calling <code>unwrap</code> with the same argument should succeed.
     *
     * @param iface a Class defining an interface.
     * @return true if this implements the interface or directly or indirectly wraps an object that does.
     * @throws java.sql.SQLException  if an error occurs while determining whether this is a wrapper
     * for an object with the given interface.
     * @since 1.6
     */
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        boolean result = false;
        try {
            Object cds = mcf.getDataSource();

            if(iface.isInstance(cds)){
                result = true;
            }else if (cds instanceof java.sql.Wrapper) {
                result = ((java.sql.Wrapper) cds).isWrapperFor(iface);
            }
        } catch (ResourceException e) {
            _logger.log(Level.WARNING, "jdbc.exc_is_wrapper", e);
            throw new SQLException(e);
        }
        return result ;
    }
}
