/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.cli.commands.monitor;

import com.sun.enterprise.cli.framework.*;
import com.sun.appserv.management.monitor.*;
import com.sun.appserv.management.monitor.statistics.*;
import javax.management.j2ee.statistics.*;
import com.sun.appserv.management.j2ee.statistics.*;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.appserv.management.util.misc.MapUtil;
import java.util.StringTokenizer;
import java.util.Map;
import java.util.Timer;
import java.util.List;
import java.util.Vector;
import java.io.File;

public class ServletMonitorTask extends MonitorTask
{
    private final String displayFormat = "%1$-8s %2$-8s %3$-8s %4$-8s %5$-8s %6$-8s "+
                                         "%7$-8s %8$-8s";

    private String appName     = null;
    private String webName     = null;
    private String servletName = null;    

    public ServletMonitorTask(final ServerRootMonitor srm, final String filter, final Timer timer,
                                             final boolean verbose, final File fileName)
        throws MonitorTaskException
    {
        super(srm, filter, timer, verbose, fileName);

        
        if (this.filter == null)
            getDefaultFilter();
        else
        {
            final StringTokenizer st = new StringTokenizer(this.filter, ":");
            if (st.countTokens() < 2 )
                throw new MonitorTaskException("Invalid filter format.  Servlet exists in either webmodule "+
                                               "or appmodule.  Please use the format: <webmodule>:<servlet> " +
                                               "or <appmodule>:<webmodule>:<servlet>.");
            else if (st.countTokens() == 2)
            {
                webName = st.nextToken();
                servletName = st.nextToken();
            }
            else {
                appName = st.nextToken();
                webName = st.nextToken();
                servletName = st.nextToken();
            }
            verifyFilterValue();
        }
        final String title = String.format("%1$50s", "Servlet Monitoring " + this.filter);
        System.out.println(title);
        displayHeader();        
    }
    

    public void run()
    {
        if (srm == null) {
            super.cancelMonitorTask();
            return;
        }
        
        Map<String,WebModuleVirtualServerMonitor> webMap = null;
        if (appName == null)
        {
            webMap = srm.getWebModuleVirtualServerMonitorMap();            
        }
        else
        {
            final Map<String,ApplicationMonitor> appMap = srm.getApplicationMonitorMap();
            if (appMap == null || appMap.size()<1) {
                cancelMonitorTask();
                return;
            }
            
            final ApplicationMonitor am = appMap.get(appName);
            webMap = am.getWebModuleVirtualServerMonitorMap();
        }
        if (webMap == null || webMap.size()<1) {
            cancelMonitorTask();
            return;
        }
        
        final ServletMonitor servletMonitor = getServletFromWebModule(webMap);
        if (servletMonitor == null) {
            cancelMonitorTask();
            return;
        }
        
        final AltServletStats servletStats = servletMonitor.getAltServletStats();
        if (verbose && counter == NUM_ROWS)
        {
            displayHeader();
            counter = 0;  //reset to 0
        }
        displayData(servletStats);
        if (verbose) counter++;
    }

    
    private ServletMonitor getServletFromWebModule(final Map<String, WebModuleVirtualServerMonitor> webMap)
    {
        final WebModuleVirtualServerMonitor webmoduleVS = webMap.get(webName);
        final Map<String, ServletMonitor> servletMap = webmoduleVS.getServletMonitorMap();
        return servletMap.get(servletName);
    }
    
    
    
    /**
     *  Returns all the possible webmodules in all the application.
     *  The return format is <app-name>:<webmodule>:<servlet>
     */
    private String[] possibleAppModuleFilters()
    {
        final Map<String,ApplicationMonitor> appModuleMap = srm.getApplicationMonitorMap();

        List<String> possibleFilters = new Vector<String>();
        if (appModuleMap != null && appModuleMap.size() > 0)
        {
                //search through all application and check for the webmodules
            final String[] appModules = MapUtil.getKeyStrings(appModuleMap);
            for (String appModule : appModules)
            {
                ApplicationMonitor am = (ApplicationMonitor)appModuleMap.get(appModule);
                    //search for the webmodule virtual server in the appmodule
                final Map<String, WebModuleVirtualServerMonitor> innerWebModuleMap = am.getWebModuleVirtualServerMonitorMap();
                possibleFilters = getServletsInWebModules(appModule, innerWebModuleMap);
            }
        }
        return (String[])possibleFilters.toArray(new String[0]);
    }
    
    
    /**
     *  Returns all the possible servlets in all the webmodules.
     *  The return format is <webmodule>:<servlet>
     */
    private String[] possibleWebModuleFilters()
    {
        final Map<String,WebModuleVirtualServerMonitor> webModuleMap = srm.getWebModuleVirtualServerMonitorMap();

        List<String> possibleFilters = getServletsInWebModules(null, webModuleMap);
        return (String[])possibleFilters.toArray(new String[0]);
    }


    private List<String> getServletsInWebModules(final String appName,
                                                 final Map<String,WebModuleVirtualServerMonitor> webModuleMap)
    {
        List<String> possibleServlets = new Vector<String>();
        final String[] webModules = MapUtil.getKeyStrings(webModuleMap);
        if (webModuleMap != null && webModuleMap.size() > 0)
        {
            for (String webModule : webModules)
            {
                final WebModuleVirtualServerMonitor webmoduleVS = webModuleMap.get(webModule);
                final Map<String,ServletMonitor> servletMap = webmoduleVS.getServletMonitorMap();
                final String[] servlets = MapUtil.getKeyStrings(servletMap);
                for (String servlet : servlets)
                {
                    if (appName == null)
                        possibleServlets.add(webModule+":"+servlet);
                    else
                        possibleServlets.add(appName+":"+webModule+":"+servlet);                        
                }
            }
        }
        return possibleServlets;
    }
    


    /**
     *  If filter option is not specified, then this method will
     *  get the default filter value.
     *  WebModuleVirtualServer is in both standalone webmodule or
     *  a webmodule in a application.  If there are more than one
     *  standalone webmodule or an application with a webmodule,
     *  this method will display the available filter selections.
     *  If there is only one webmodule then that is the default
     *  filter value.
     */
    private void getDefaultFilter() throws MonitorTaskException
    {
        String[] appModuleFilters = possibleAppModuleFilters();
        String[] webModuleFilters = possibleWebModuleFilters();        
            
        if (webModuleFilters.length < 1 && appModuleFilters.length < 1)
        {
            throw new MonitorTaskException("No value to monitor.");
        }
        else if (webModuleFilters.length == 1 && appModuleFilters.length < 1)
        {
            final String temp = webModuleFilters[0];
            final StringTokenizer st = new StringTokenizer(temp, ":");
            webName = st.nextToken();
            servletName = st.nextToken();
        }
        else if (webModuleFilters.length < 1 && appModuleFilters.length == 1)
        {
            final String temp = appModuleFilters[0];
            final StringTokenizer st = new StringTokenizer(temp, ":");
            appName = st.nextToken();
            webName = st.nextToken();
            servletName = st.nextToken();            
        }
        else
        {
            final String msg = getAvailableFilterMessage(appModuleFilters, webModuleFilters);
            throw new MonitorTaskException(msg);
        }
    }


    private String getAvailableFilterMessage(final String[] appModuleFilters,
                                             final String[] webModuleFilters)
    {
        StringBuffer sb = new StringBuffer();
        sb.append("There are more than one monitoring elements. ");
        sb.append("Please consider using the filter option.\n");
        if (webModuleFilters.length > 0)
        {
            sb.append("The following are available elements to monitor for webmodules <webmodule>:<servlet> \n");
            for (String webmodule: webModuleFilters)
            {
                sb.append("    ");
                sb.append(webmodule);
                sb.append("\n");
            }
        }
        if (appModuleFilters.length > 0)
        {
            sb.append("The following are available elements to monitor for appmodules <appmodule>:<webmodule>:<servlet> \n");
            for (String appmodule: appModuleFilters)
            {
                sb.append("    ");
                sb.append(appmodule);
                sb.append("\n");
            }
        }
        return sb.toString();
    }
    


    private void verifyFilterValue() throws MonitorTaskException
    {
        if (appName != null)
        {
            final Map<String,ApplicationMonitor> appModuleMap = srm.getApplicationMonitorMap();
            if (appModuleMap.containsKey(appName))
            {
                final ApplicationMonitor am = appModuleMap.get(appName);
                final Map<String, WebModuleVirtualServerMonitor> innerWebModuleMap = am.getWebModuleVirtualServerMonitorMap();
                final boolean exist = checkIfServletExistInWebModule(innerWebModuleMap);
                if (!exist)
                    throw new MonitorTaskException(servletName + " does not exist in "+appName+":"+webName+".");
            }
            else {
                throw new MonitorTaskException(appName + " does not exist.");
            }
        }
        else
        {
            final Map<String,WebModuleVirtualServerMonitor> webModuleMap = srm.getWebModuleVirtualServerMonitorMap();
            final boolean exist = checkIfServletExistInWebModule(webModuleMap);
            if (!exist)
                    throw new MonitorTaskException(servletName + " does not exist in "+webName+".");
        }
    }
    

        /**
         * returns true if servlet exists in webmodule
         * @throws MonitorTaskException if webmodlue is invalid.
         */
    private boolean checkIfServletExistInWebModule(final Map<String,WebModuleVirtualServerMonitor> webModuleMap)
        throws MonitorTaskException
    {
        if (!webModuleMap.containsKey(webName))
        {
            if (appName == null)
                throw new MonitorTaskException(webName + " does not exist.");
            else
                throw new MonitorTaskException(webName + " does not exist in " + appName+ ".");                
        }
        else {
            final WebModuleVirtualServerMonitor webmoduleVS = webModuleMap.get(webName);
            final Map<String,ServletMonitor> servletMap = webmoduleVS.getServletMonitorMap();
            if (!servletMap.containsKey(servletName))
                return false;
            else
                return true;
        }
    }
    
    
    private void displayHeader()
    {
        final String header = String.format(displayFormat,
                                            "ec","mt","pt","rc","current","maxtm","mintm","totaltm");
        System.out.println(header);
        if (fileName != null)
        {
            writeToFile("ErrorCount,MaxTime,ProcessingTime,RequestCount,ServiceTime:count,"+
                        "ServiceTime:maxtime,ServiceTime:mintime,ServiceTime:totaltime");
        }
    }

    
    private void displayData(final AltServletStats stat)
    {
        final String data = String.format(displayFormat,
                                          stat.getErrorCount().getCount(),
                                          stat.getMaxTime().getCount(),
                                          stat.getProcessingTime().getCount(),
                                          stat.getRequestCount().getCount(),
                                          stat.getServiceTime().getCount(),
                                          stat.getServiceTime().getMaxTime(),
                                          stat.getServiceTime().getMinTime(),
                                          stat.getServiceTime().getTotalTime());
        
        System.out.println(data);
        if (fileName != null)
        {
            final String fileData = String.format("%1$s,%2$s,%3$s,%4$s,%5$s,%6$s,%7$s,%8$s",
                                                  stat.getErrorCount().getCount(),
                                                  stat.getMaxTime().getCount(),
                                                  stat.getProcessingTime().getCount(),
                                                  stat.getRequestCount().getCount(),
                                                  stat.getServiceTime().getCount(),
                                                  stat.getServiceTime().getMaxTime(),
                                                  stat.getServiceTime().getMinTime(),
                                                  stat.getServiceTime().getTotalTime());
            writeToFile(fileData);
        }
    }

    
    public void displayDetails()
    {
        System.out.println("*****************************************************************************************");
        System.out.println("* ec      = Cumulative value of the error count. The error count represents the         *");
        System.out.println("            number of cases where the response code was greater than or equal to 400.   *");
        System.out.println("* mt      = Longest response time for a request - not a cumulative value, but the       *");
        System.out.println("*           largest response time from among the response times.                        *");
        System.out.println("* pt      = Cumulative value of the times taken to process each request. The processing *");
        System.out.println("*           time is the average of request processing times over the request count.     *");
        System.out.println("* rc      = Cumulative number of the requests processed so far.                         *");
        System.out.println("* current = Current execution time of the servlet's service method as TimeStatistic.    *");
        System.out.println("* maxtm   = Maximum execution time of the servlet's service method as TimeStatistic.    *");
        System.out.println("* mintm   = Minimum execution time of the servlet's service method as TimeStatistic.    *");
        System.out.println("* totaltm = Total execution time of the servlet's service method as TimeStatistic.      *");
        System.out.println("*****************************************************************************************");        
    }


}
    
    

