/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.admin.wsmgmt.transform;

import com.sun.enterprise.admin.wsmgmt.config.spi.Constants;
import com.sun.enterprise.admin.wsmgmt.config.spi.TransformationRule;
import com.sun.enterprise.admin.wsmgmt.filter.spi.Filter;
import com.sun.enterprise.admin.wsmgmt.filter.spi.FilterContext;

import com.sun.enterprise.admin.wsmgmt.config.spi.WebServiceConfig;

import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.concurrent.locks.Lock;

import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;
import com.sun.enterprise.util.i18n.StringManager;



/**
 * Filter that can implement XSLT transformations.
 */
public class TransformFilter implements Filter {

    /**
     * Public Constructor.
     *
     * @param appId   name of the application
     * @param endpoint   end point name for which stats are collected
     */
    public TransformFilter(String appId, String endpoint) {
        _applicationId = appId;
        _endpointId    = endpoint;
        _rwl = new ReentrantReadWriteLock();
        _readLock = _rwl.readLock();
        _writeLock = _rwl.writeLock();
    }

    /**
     * Public Constructor.
     *
     * @param appId   name of the application
     * @param endpoint   end point name for which stats are collected
     */
    public TransformFilter(String appId, WebServiceConfig wsc) 
        throws TransformException {
        _applicationId = appId;
        _endpointId    = wsc.getName();
        TransformationRule[] reqRules = wsc.getRequestTransformationRule();
        if ( reqRules != null) {
            reqChain = new FilterChain();
            reqChain.addFilter(reqRules, false);
        } else {
            reqChain = null;
        }
        TransformationRule[] resRules = wsc.getResponseTransformationRule();
        if (resRules != null) {
            resChain = new FilterChain();
            resChain.addFilter(resRules, true);
        } else {
            resChain = null;
        }
        _rwl = new ReentrantReadWriteLock();
        _readLock = _rwl.readLock();
        _writeLock = _rwl.writeLock();
    }

    // XXX optimize these resetXXXChain, rather than creating the new chain
    // modify the existing RequestChain. 

    public void resetRequestChain(TransformationRule[] tRules) throws
    TransformException {
        _writeLock.lock();  
        try {
            reqChain = new FilterChain();
            reqChain.addFilter(tRules, false);
        } finally {
           _writeLock.unlock();
        }
    }

    public void resetResponseChain(TransformationRule[] tRules)  throws
    TransformException {
        _writeLock.lock();  
        try {
            resChain = new FilterChain();
            resChain.addFilter(tRules, true);
        } finally {
           _writeLock.unlock();
        }
    }

    /**
     * Returns the unique name for this filter
     */
    public String getName() {
        return (NAME_PREFIX + _applicationId + DELIM + _endpointId);
    }

    /**
     * Returns the unique name for this filter
     */
    static public String getName(String appId, WebServiceConfig wsc) {
        return (NAME_PREFIX + appId + DELIM + wsc.getName());
    }

    /**
     * Invoke the filter.
     * 
     * @param  stage   stage of the execution
     * @param  endpoint  name of the endpoint
     * @param  context  filter context 
     */
    public void process(String stage, String endpoint, FilterContext context) {

        _readLock.lock();
        try {
            // SOAP request
            if ( stage.equals(Filter.PROCESS_REQUEST) ) {
                // optmize this XXX
                if ( reqChain != null) {
                        reqChain.process(context);
                }
            // SOAP response
            } else if ( stage.equals(Filter.PROCESS_RESPONSE) ) {
                if ( resChain != null) {
                        resChain.process(context);
                }
            }
        } catch(Exception e) {
            // log a warning
            e.printStackTrace();
            String msg = _stringMgr.getString("transform_failed", endpoint);
            _logger.log(Level.INFO, msg, e.getMessage());
        } finally {
            _readLock.unlock();
        }

    }

    // -- PRIVATE - VARIABLES -------------------------
    private String _applicationId            = null;
    private String _endpointId               = null;
    private static final String DELIM        = "#";
    private static final String NAME_PREFIX  = "TRANSFORMFILTER_";
    private FilterChain reqChain = null;
    private FilterChain resChain = null;
    private ReentrantReadWriteLock _rwl = null;
    private Lock _readLock = null;
    private Lock _writeLock = null;
    private static final Logger _logger = 
        Logger.getLogger(LogDomains.ADMIN_LOGGER);
    private static final StringManager _stringMgr = 
        StringManager.getManager(TransformFilter.class);


}
