/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.annotation.factory;

import java.util.HashSet;
import java.util.Set;
import java.util.logging.Level;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.InputStream;
import java.io.IOException;

import com.sun.enterprise.deployment.annotation.Scanner;
import com.sun.enterprise.deployment.annotation.AnnotationProcessor;
import com.sun.enterprise.deployment.annotation.AnnotationHandler;
import com.sun.enterprise.deployment.annotation.impl.AnnotationProcessorImpl;
import com.sun.enterprise.deployment.annotation.impl.AnnotationUtils;
/**
 * The Factory is responsible for initializing a ready to use AnnotationProcessor. 
 * 
 * @author Jerome Dochez
 */
public class Factory {
    
    protected static AnnotationProcessorImpl systemProcessor=null;
   
    private static Set<String> skipAnnotationClassList = null;
    private static final String SKIP_ANNOTATION_CLASS_LIST_URL =
        "skip-annotation-class-list";

    /** we do no Create new instances of Factory */
    protected Factory() {
    }

    private static void init() {
    }

    /**
     * Return a fully initialized AnnotationProcessor where all the system
     * annotation handlers have been registered.
     * @return initialized AnnotationProcessor instance
     */ 
    public static AnnotationProcessor getAnnotationProcessor() {
        
        init();

        // we use a delegation system. The System annotation processor is 
        // always the same instance. The user gets handed over a new 
        // instance of the annotation processor which no annotation handler
        // registered. The user uses that instance to register it's own (if 
        // any) annotation handler. 
        AnnotationProcessorImpl processor = new AnnotationProcessorImpl();
        processor.setDelegate(systemProcessor);
        return processor;
    }
    
    /**
     * Return a empty AnnotationProcessor with no annotation handlers registered
     * @return initialized AnnotationProcessor instance
     */ 
    public static AnnotationProcessor getDefaultAnnotationProcessor() {
        return new AnnotationProcessorImpl();        
    }    

    // initialize the list of class files we should skip annotation processing
    private synchronized static void initSkipAnnotationClassList() {
        if (skipAnnotationClassList == null) {
            skipAnnotationClassList = new HashSet<String>();
            InputStream is = null;
            try {
                ClassLoader cl = Thread.currentThread().getContextClassLoader();
                is = cl.getResourceAsStream(SKIP_ANNOTATION_CLASS_LIST_URL);
                BufferedReader bf =
                    new BufferedReader(new InputStreamReader(is));
                String className;
                while ( (className = bf.readLine()) != null ) {
                    skipAnnotationClassList.add(className.trim());
                }
            } catch (IOException ioe) {
                AnnotationUtils.getLogger().log(Level.WARNING, 
                    ioe.getMessage(), ioe);
            } finally {
                if (is != null) {
                    try {
                        is.close(); 
                    } catch (IOException ioe2) {
                        // ignore
                    }
                }
            }
        }
    }

    // check whether a certain class can skip annotation processing
    public static boolean isSkipAnnotationProcessing(String cName) {
        if (skipAnnotationClassList == null) {
            initSkipAnnotationClassList();
        }
        return skipAnnotationClassList.contains(cName); 
    }

}
