/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.addons;

import com.sun.appserv.addons.AddonVersion;
import com.sun.appserv.addons.AddonException;

/**
 * Parse the given addon name into majr, minor and patch levels
 * and return those component versions to the callers.
 *
 * @since 9.1
 * @authod sreenivas.munnangi@sun.com
 */
public class AddonVersionImpl implements AddonVersion {

    private String addonName = null;
    private String namePart = null;
    private String version = null;
    private int major = 0;
    private int minor = 0;
    private int patch = 0;

    private final String DEFAULT_VERSION = "00_00_00";
    private final String VERSION_PATTERN = ".*_[0-9][0-9]_[0-9][0-9]_[0-9][0-9]";
    private final String VERSION_SEPARATOR = "_";

    /**
     * Constructor
     */
    AddonVersionImpl(String addonName) throws AddonException {
        this.addonName = addonName;
        if (addonName.matches(VERSION_PATTERN)) {
            version = addonName.substring((addonName.length() - DEFAULT_VERSION.length()));
            namePart = addonName.substring(0, (addonName.length() - (DEFAULT_VERSION.length() + 1)));
        } else {
            version = DEFAULT_VERSION;
            namePart = addonName;
        }
        parseVersion(version);
    }

    private void parseVersion(String versionPart) throws AddonException {
        String [] strArr = versionPart.split(VERSION_SEPARATOR);
        setMajor(strArr[0]);
        setMinor(strArr[1]);
        setPatch(strArr[2]);
    }

    /**
     * Get the full version as string, for example.
     * If the addon is named as am_components_installer_01_02_03.jar
     * then the version output will be "01_02_03" in String format.
     * @return String version
     */
    public String getVersion() throws AddonException {
        return version;
    }

    /**
     * Get the majr version, for example.
     * If the addon is named as am_components_installer_01_02_03.jar
     * then the value of 1 will be returned.
     * @return int major
     */
    public int getMajor() throws AddonException {
        return this.major;
    }

    private void setMajor(String major) throws AddonException {
        try {
            this.major = (Integer.valueOf(major)).intValue();
        } catch (Exception e) {
            throw new AddonException(e);
        }
    }

    /**
     * Get the minor version, for example.
     * If the addon is named as am_components_installer_01_02_03.jar
     * then the value of 2 will be returned.
     * @return int minor
     */
    public int getMinor() throws AddonException {
        return this.minor;
    }

    private void setMinor(String minor) throws AddonException {
        try {
            this.minor = (Integer.valueOf(minor)).intValue();
        } catch (Exception e) {
            throw new AddonException(e);
        }
    }

    /**
     * Get the patch version, for example.
     * If the addon is named as am_components_installer_01_02_03.jar
     * then the value of 3 will be returned.
     * @return int patch
     */
    public int getPatch() throws AddonException {
        return this.patch;
    }

    private void setPatch(String patch) throws AddonException {
        try {
            this.patch = (Integer.valueOf(patch)).intValue();
        } catch (Exception e) {
            throw new AddonException(e);
        }
    }

    protected String getName() {
        return addonName;
    }

    protected String getNamePart() {
        return namePart;
    }

    public String toString() {
        StringBuffer sb = new StringBuffer("");
        try {
        sb.append("Version details for the AddOn " + addonName + ":"); 
        sb.append("version = " + getVersion() + ":");
        sb.append("major = " + getMajor() + ":");
        sb.append("minor = " + getMinor() + ":");
        sb.append("patch = " + getPatch());
        } catch (Exception e) {
        }
        return sb.toString();
    }

    protected boolean isHigher(AddonVersionImpl newVersion) 
        throws AddonException {

        if (newVersion.getMajor() > getMajor()) {
            return true;
        } else if (newVersion.getMajor() < getMajor()) {
            return false;
        }

        if (newVersion.getMinor() > getMinor()) {
            return true;
        } else if (newVersion.getMinor() < getMinor()) {
            return false;
        }

        if (newVersion.getPatch() > getPatch()) {
            return true;
        }

        return false;
    }

    protected boolean isLower(AddonVersionImpl newVersion) 
        throws AddonException {

        if (newVersion.getMajor() < getMajor()) {
            return true;
        } else if (newVersion.getMajor() > getMajor()) {
            return false;
        }

        if (newVersion.getMinor() < getMinor()) {
            return true;
        } else if (newVersion.getMinor() > getMinor()) {
            return false;
        }

        if (newVersion.getPatch() < getPatch()) {
            return true;
        }

        return false;
    }

}
