/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.jbi.serviceengine.comm;

import com.sun.enterprise.jbi.serviceengine.core.JavaEEServiceEngineContext;
import com.sun.enterprise.jbi.serviceengine.util.DOMUtil;
import com.sun.enterprise.jbi.serviceengine.util.JBIConstants;
import com.sun.enterprise.jbi.serviceengine.util.soap.EndpointMetaData;
import com.sun.enterprise.jbi.serviceengine.util.soap.SOAPConstants;
import com.sun.logging.LogDomains;
import com.sun.xml.bind.api.Bridge;
import com.sun.xml.ws.api.message.HeaderList;
import com.sun.xml.ws.api.message.Message;
import com.sun.enterprise.jbi.serviceengine.util.DOMStreamReader;
import com.sun.enterprise.jbi.serviceengine.util.StAXSource;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.jbi.messaging.Fault;
import javax.jbi.messaging.NormalizedMessage;
import javax.wsdl.Part;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.namespace.QName;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ContentHandler;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;

/**
 *
 * This class is used to unwrap the incoming JBI request message when the
 * Java EE Service Engine is the Provider.
 *
 * This class unwraps and extracts the 'payload' from the incomoming
 * request message which is in normalized form.
 *
 * The current implentation supports the following WSDL1.1 SOAP binding styles :
 *
 *      1. Wrapped Document/literal
 *      2. RPC/Literal
 *
 * The characterstics of the Wrapped Document/literal style are
 *
 *  1. The input message has a single part.
 *  2. The part is an element.
 *  3. The element has the same name as the operation.
 *  4. The element's complex type has no attributes.
 *
 * For wrapped document/literal, the incoming request message looks like :
 *
 * <?xml version="1.0" encoding="UTF-8"?>
 * <jbi:message
 *          xmlns:msgns="http://example.web.service/Calculator"
 *          name="add"
 *          type="msgns:add"
 *          version="1.0"
 *          xmlns:jbi="http://java.sun.com/xml/ns/jbi/wsdl-11-wrapper">
 *	<jbi:part>
 *		<ns2:add xmlns:ns2="http://example.web.service/Calculator">
 *			<arg0>1.0</arg0>
 *			<arg1>2.0</arg1>
 *		</ns2:add>
 *	</jbi:part>
 * </jbi:message>
 *
 * and the payLoad node for this is :
 *
 * <ns2:add xmlns:ns2="http://example.web.service/Calculator">
 *      <arg0>1.0</arg0>
 *      <arg1>2.0</arg1>
 * </ns2:add>
 *
 * For RPC/literal, the incoming request message looks like this :
 *
 * <?xml version="1.0" encoding="UTF-8"?>
 * <jbi:message
 *          xmlns:msgns="http://example.web.service/Calculator"
 *          name="add"
 *          type="msgns:add"
 *          version="1.0"
 *          xmlns:jbi="http://java.sun.com/xml/ns/jbi/wsdl-11-wrapper">
 *	<jbi:part><int_1>1</int_1></jbi:part>
 *	<jbi:part><int_2>2</int_2></jbi:part>
 * </jbi:message>
 *
 * and the payLoad node for this is:
 *
 * <ns2:add xmlns:ns2="http://example.web.service/Calculator">
 *      <int_1>1</int_1>
 *      <int_2>2</int_2>
 * </ns2:add>
 *
 * The styles which are not supported by JAX-WS 2.0 are:
 *      RPC/encoded
 *      Document/encoded
 *
 * The style(s) which are treated/converted as wrapped document/literal by JAX-WS tools are
 *      document/literal
 *
 * @author bhavanishankar@dev.java.net
 */

public final class UnWrappedMessage extends Message
        implements JBIConstants, SOAPConstants {
    
    private static Logger logger =
            LogDomains.getLogger(LogDomains.SERVER_LOGGER);
    
    private String payloadLocalName;
    private String payloadNamespaceURI;
    private Source payLoadAsSource;
    private XMLStreamReader payLoadAsStreamReader;
    private ByteArrayOutputStream payLoadAsBaos;
    private boolean isFault;
    
    /**
     * Search for the payLoad node in the given src and initialize
     * payLoadAsSource and payloadAsStreamReader.
     *
     * If the payLoad node is not found in the given src then throw an
     * Exception informing the user not to use this class.
     *
     * @param src JBI inbound request message
     * @param operation operation to invoke on the endpoint
     * @see NMRServerConnection
     */
    public UnWrappedMessage(NormalizedMessage normalizedMessage,
            String operationName,
            EndpointMetaData emd,
            boolean isResponse) throws Exception {
        if(normalizedMessage instanceof Fault) {
            isFault = true;
        }
        if(JavaEEServiceEngineContext.getInstance().isServiceMix()) {
            logger.log(Level.FINE, "Skipping the unwrapping...");
            setPayLoad(normalizedMessage.getContent());
            return;
        }
        String bindingStyle = emd.getBindingStyle(operationName);
        Source src = normalizedMessage.getContent();
        String s = (src instanceof DOMSource) ? toString(src) : "StreamSource";
        logger.log(Level.FINE, "UnWrappedMessage :: bindingStyle = " + bindingStyle
                + ", operationName = " + operationName
                + ", received message = " + s
                );
        
        QName wrappedMessageType = isResponse
                //? emd.getOutputMessage(operationName).getQName()
                //: emd.getInputMessage(operationName).getQName();
                ? new QName(emd.getOutputMessage(operationName).getQName().getNamespaceURI(), operationName + "Response")
                : new QName(emd.getInputMessage(operationName).getQName().getNamespaceURI(), operationName);
        
        //StreamSource ss = getStreamSource(src);
        if(RPC_STYLE.equalsIgnoreCase(bindingStyle)) {
            javax.wsdl.Message message = isResponse
                    ? emd.getOutputMessage(operationName)
                    : emd.getInputMessage(operationName);
            List<Part> orderedParts = message.getOrderedParts(null);
            new RPCStyleUnWrapper().unwrap(src, wrappedMessageType, orderedParts);
        } else {
            new DocumentStyleUnWrapper().unwrap(src);
        }
        if(payLoadAsStreamReader == null) {
            throw new Exception("Direct invocation failed. Unable to unwrap the request.");
        }
    }
    
    public boolean isFault() {
        return isFault;
    }
    
    public String getPayloadLocalPart() {
        return payloadLocalName;
    }
    
    public String getPayloadNamespaceURI() {
        return payloadNamespaceURI;
    }
    
    public boolean hasPayload() {
        return payLoadAsStreamReader == null;
    }
    
    public boolean hasHeaders() {
        return false;
    }
    
    public HeaderList getHeaders() {
        return new HeaderList();
    }
    
    public Message copy() {
        return null;
    }
    
    public Source readEnvelopeAsSource() {
        return null;
    }
    
    public XMLStreamReader readPayload() throws XMLStreamException {
        logger.log(Level.FINE, "UnWrappedMessage :: readPayLoad()");
        return payLoadAsStreamReader;
    }
    
    public Source readPayloadAsSource() {
        logger.log(Level.FINE, "UnWrappedMessage :: readPayLoadAsSource()");
        return payLoadAsSource;
    }
    
    public SOAPMessage readAsSOAPMessage() throws SOAPException {
        String methodSig =
                "\ncom.sun.enterprise.jbi.serviceengine.comm.UnWrappedMessage" +
                "::readAsSOAPMessage()";
        String usedWith = System.getProperty(USED_WITH);
        if(usedWith == null || usedWith.indexOf(USED_WITH_JMAC_PROVIDER) == -1) {
            throw new SOAPException(
                    methodSig + " operation is not supported." +
                    "\nSet this system property to workaround this issue : " +
                    "com.sun.enterprise.jbi.se.usedwith=jmacprovider");
        }
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            XMLStreamWriter writer = XMLOutputFactory.newInstance().createXMLStreamWriter(baos);
            writer.writeStartDocument();
            writer.writeStartElement(SOAP_ENVELOPE);
            writer.writeAttribute("xmlns:" + SOAP_PREFIX, SOAP_NAMESPACE);
            writer.writeEmptyElement(SOAP_HEADER);
            writer.writeStartElement(SOAP_BODY);
            if(payLoadAsBaos == null) {
                if(payLoadAsSource instanceof DOMSource) {
                    Node payLoadNode = ((DOMSource)payLoadAsSource).getNode();
                    DOMUtil.UTIL.writeNode(payLoadNode, writer);
                } else {
                    DOMUtil.UTIL.writeNode(payLoadAsStreamReader, writer);
                }
            } else {
                baos.write(">".getBytes());
                baos.write(payLoadAsBaos.toByteArray());
            }
            writer.writeEndElement();
            writer.writeEndElement();
            writer.writeEndDocument();
            ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
            
            SOAPMessage message = MessageFactory.newInstance().createMessage(null, bais);
            if(logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, methodSig + " :: SOAPMessage = " + toString(message));
            }
            return message;
        } catch(Exception ex) {
            throw new SOAPException(methodSig + ex.getMessage());
        }
    }
    
    public <T> T readPayloadAsJAXB(Unmarshaller unmarshaller) throws JAXBException {
        return (T)unmarshaller.unmarshal(payLoadAsSource);
    }
    
    public <T> T readPayloadAsJAXB(Bridge<T> bridge) throws JAXBException {
        return bridge.unmarshal(payLoadAsSource);
    }
    
    public void writePayloadTo(XMLStreamWriter sw) throws XMLStreamException {
        throw new XMLStreamException("Operaion is not supported.");
    }
    
    public void writeTo(XMLStreamWriter sw) throws XMLStreamException {
        throw new XMLStreamException("Operaion is not supported.");
    }
    
    public void writeTo(ContentHandler contentHandler, ErrorHandler errorHandler) throws SAXException {
        throw new SAXException("Operaion is not supported.");
    }
    
    // Methods for debugging purposes.
    
    public static String toString(SOAPMessage soapMessage) {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            soapMessage.writeTo(baos);
            return baos.toString();
        } catch(Exception e) {
            return e.getMessage();
        }
    }
    
    
    public static String toString(Node n) {
        return toString(new DOMSource(n));
    }
    
    public static String toString(XMLStreamReader reader) {
        return toString(new StAXSource(reader, true));
    }
    
    public static String toString(Source s) {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            StreamResult sr = new StreamResult(baos);
            TransformerFactory.newInstance().newTransformer().transform(s, sr);
            return baos.toString();
        } catch(Exception ex) {
            ex.printStackTrace();
            return null;
        }
    }
    
    public StreamSource getStreamSource(Source src) {
        try {
            if(src instanceof StreamSource) {
                return (StreamSource)src;
            }
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            StreamResult result = new StreamResult(baos);
            TransformerFactory.newInstance().newTransformer().transform(src, result);
            ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
            StreamSource ss = new StreamSource(bais);
            return ss;
        } catch(Exception ex) {
            ex.printStackTrace();
            return null;
        }
        
    }
    
    public void setPayLoad(Source s) throws Exception {
        if(s instanceof DOMSource) {
            setPayLoad(((DOMSource)s).getNode());
        } else if(s instanceof StreamSource) {
            XMLStreamReader reader = XMLInputFactory.newInstance().
                    createXMLStreamReader(((StreamSource)s).getInputStream());
            setPayLoad(reader);
        }  else if(s instanceof SAXSource) {
            XMLStreamReader reader = XMLInputFactory.newInstance().
                    createXMLStreamReader(((SAXSource)s).getInputSource().getByteStream());
            setPayLoad(reader);
        } else {
            logger.log(Level.WARNING, "UnWrappedMessage :: Transforming the input message to DOM");
            Transformer t = TransformerFactory.newInstance().newTransformer();
            DOMResult result = new DOMResult();
            t.transform(s, result);
            setPayLoad(result.getNode());
        }
    }
    
    public void setPayLoad(Node n) {
        if(n.getNodeType() == Node.DOCUMENT_NODE) {
            n = n.getFirstChild();
        }
        payloadLocalName = n.getLocalName();
        payloadNamespaceURI = n.getNamespaceURI();
        payLoadAsSource = new DOMSource(n);
        payLoadAsStreamReader = new DOMStreamReader(n);
        printPayLoad("");
    }
    
    public void setPayLoad(XMLStreamReader reader) throws Exception {
        if(reader.getEventType() == XMLStreamReader.START_DOCUMENT) {
            reader.next();
        }
        payloadLocalName =  reader.getLocalName();
        payloadNamespaceURI = reader.getNamespaceURI();
        payLoadAsSource = new StAXSource(reader, true); // StAXSource will be available in JDK6.
        payLoadAsStreamReader = reader;
        printPayLoad("");
    }
    
    public void printPayLoad(String message) {
        if(!logger.isLoggable(Level.FINE)) {
            return;
        }
        StringBuffer msg = new StringBuffer("\n\n");
        if(payLoadAsSource instanceof DOMSource) {
            Node n = ((DOMSource)payLoadAsSource).getNode();
            msg.append("Unwrapped message " + UnWrappedMessage.toString(n));
        }
        msg
                .append(message)
                .append("\n\npayLoadLocalName = ")
                .append(payloadLocalName)
                .append("\npayLoadNamespaceURI = ")
                .append(payloadNamespaceURI)
                .append("\n\n");
        
        logger.log(Level.FINE, msg.toString());
    }
// END :: Methods for debugging purposes.
    
    class DocumentStyleUnWrapper {
        
        void unwrap(Source wrappedMessage) throws Exception {
            if(wrappedMessage instanceof DOMSource) { // most cases it will be DOMSource
                Node n = ((DOMSource)wrappedMessage).getNode();
                if(isFault) {
                    setPayLoad(n);
                } else {
                    unwrap(getOwnerDocument(n));
                }
            } else if(wrappedMessage instanceof StreamSource) {
                XMLStreamReader reader = XMLInputFactory.newInstance().
                        createXMLStreamReader(((StreamSource)wrappedMessage).getInputStream());
                if(isFault) {
                    setPayLoad(reader);
                } else {
                    unwrap(reader);
                }
            } else if(wrappedMessage instanceof SAXSource) {
                XMLStreamReader reader = XMLInputFactory.newInstance().
                        createXMLStreamReader(((SAXSource)wrappedMessage).getInputSource().getByteStream());
                if(isFault) {
                    setPayLoad(reader);
                } else {
                    unwrap(reader);
                }
            } else {
                logger.log(Level.WARNING, "UnWrappedMessage :: Transforming the input message to DOM");
                Transformer t = TransformerFactory.newInstance().newTransformer();
                DOMResult result = new DOMResult();
                t.transform(wrappedMessage, result);
                if(isFault) {
                    setPayLoad(result.getNode());
                } else {
                    unwrap(getOwnerDocument(result.getNode()));
                }
            }
        }
        
        /**
         * Searches for the payLoad node in the given node and makes
         * payLoadAsSource and payLoadAsStreamReader point to the 'payload' node.
         *
         * If the payLode node is not found in the given node, then payLoadAsSource
         * and payLoadAsStreamReader will remain null.
         *
         * @param n node representing a JBI inbound request message
         * @param operation operation to be invoked on the endpoint
         */
        private void unwrap(Node n) {
            String methodSig = "UnWrappedMessage$DocumentStyleUnwrapper :: " +
                    "unwrap(Node, QName) : ";
            while(n.hasChildNodes()) {
                n = n.getFirstChild();
                try {
                    if(logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE, methodSig + " currentNode = " + UnWrappedMessage.toString(n));
                    }
                    //String nodeName = n.getPrefix() + ":" + n.getLocalName();
                    if(isJBINode(n, WRAPPER_PART_LOCALNAME)) {
                        setPayLoad(n.getFirstChild());
                        break;
                    }
                } catch(Exception ex) {
                    logger.log(Level.SEVERE, ex.getMessage(), ex);
                }
            }
        }
        
        /**
         * Searches for the payLoad in the given reader and makes
         * payLoadAsSource and payLoadAsStreamReader point to the 'payload' node.
         *
         * If the payLode is not found in the given reader, then payLoadAsSource
         * and payLoadAsStreamReader will remain null.
         *
         * @param reader reader pointing to a JBI inbound request message
         * @param operation operation to be invoked on the endpoint
         */
        private void unwrap(XMLStreamReader reader) {
            String methodSig = "UnWrappedMessage$DocumentStyleUnwrapper :: " +
                    "unwrap(XMLStreamReader, QName) : ";
            try {
                while(reader.hasNext()) {
                    reader.next();
                    try {
                        //String nodeName = reader.getPrefix() + ":" + reader.getLocalName();
                        if(isJBINode(reader, WRAPPER_PART_LOCALNAME)) {
                            reader.next();
                            setPayLoad(reader);
                            break;
                        }
                    } catch(Exception ex) {
                        logger.log(Level.SEVERE, ex.getMessage(), ex);
                    }
                }
            } catch(Exception ex) {
                logger.log(Level.SEVERE, ex.getMessage(), ex);
            }
        }
    }
    
    class RPCStyleUnWrapper {
        
        void unwrap(Source wrappedMessage,
                QName wrappedMessageType,
                List<Part> parts
                ) throws Exception {
            if(wrappedMessage instanceof DOMSource) { // most cases it will be DOMSource
                Node n = ((DOMSource)wrappedMessage).getNode();
                if(isFault) {
                    setPayLoad(n);
                } else {
                    unwrap(getOwnerDocument(n), wrappedMessageType, parts);
                }
            } else if(wrappedMessage instanceof StreamSource) {
                XMLStreamReader reader = XMLInputFactory.newInstance().
                        createXMLStreamReader(((StreamSource)wrappedMessage).getInputStream());
                if(isFault) {
                    setPayLoad(reader);
                } else {
                    unwrap(reader, wrappedMessageType, parts);
                }
            } else if(wrappedMessage instanceof SAXSource) {
                XMLStreamReader reader = XMLInputFactory.newInstance().
                        createXMLStreamReader(((SAXSource)wrappedMessage).getInputSource().getByteStream());
                if(isFault) {
                    setPayLoad(reader);
                } else {
                    unwrap(reader, wrappedMessageType, parts);
                }
            } else {
                logger.log(Level.WARNING, "UnWrappedMessage :: Transforming the input message to DOM");
                Transformer t = TransformerFactory.newInstance().newTransformer();
                DOMResult result = new DOMResult();
                t.transform(wrappedMessage, result);
                if(isFault) {
                    setPayLoad(result.getNode());
                } else {
                    unwrap(getOwnerDocument(result.getNode()), wrappedMessageType, parts);
                }
            }
        }
        
        /**
         *
         * Unwraps the incoming JBI request message and sets
         * payLoadAsSource and payLoadAsStreamReader to hold the 'payload'.
         *
         * @param wrappedDocument XMLStreamReader or Node representing the JBI
         * request message which is in the following form:
         *
         * <jbi:message
         *          xmlns:msgns="http://example.web.service/Calculator"
         *          name="add"
         *          type="msgns:add"
         *          version="1.0"
         *          xmlns:jbi="http://java.sun.com/xml/ns/jbi/wsdl-11-wrapper">
         * 	<jbi:part><int_1>1</int_1></jbi:part>
         * 	<jbi:part><int_2>2</int_2></jbi:part>
         * </jbi:message>
         * @param payLoadName Name of the operation to be invoked on the endpoint
         * @param payLoadNsUri Target name space URI which is unique for a webservice.
         * @return Sets payLoadAsSource and payLoadAsStreamReader to the payload which is
         *
         * <ns2:add xmlns:ns2="http://example.web.service/Calculator">
         *      <int_1>1</int_1>
         *      <int_2>2</int_2>
         * </ns2:add>
         */
        private void unwrap(Object wrappedDocument,
                QName wrappedMessageType,
                List<Part> parts) {
            String methodSig = "UnWrappedMessage$RPCStyleUnwrapper :: " +
                    "unwrap(Object, String, String, ServiceEngineEndpoint) : ";
            try {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                XMLStreamWriter writer = XMLOutputFactory.newInstance().createXMLStreamWriter(baos);
                /**
                 * Extract the payLoad content from wrappedDocument and write it
                 * to writer. This does not involve creating or importing a DOM node.
                 */
                writer.writeStartElement(DEFAULT_OPERATION_PREFIX + ":" + wrappedMessageType.getLocalPart());
                writer.writeAttribute(DEFAULT_XML_NS_SCHEME + ":" + DEFAULT_OPERATION_PREFIX,
                        wrappedMessageType.getNamespaceURI());
                if(wrappedDocument instanceof Node) {
                    writeJBIParts((Node)wrappedDocument, writer, parts);
                } else if(wrappedDocument instanceof XMLStreamReader) {
                    writeJBIParts((XMLStreamReader)wrappedDocument, writer, parts);
                }
                writer.writeEndElement();
                writer.flush();
                /**
                 * Create payLoadAsStreamReader and payLoadAsSource from the
                 * content available in ByteArrayOutputStream.
                 */
                ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
                payLoadAsBaos = baos;
                payloadLocalName = wrappedMessageType.getLocalPart();
                payloadNamespaceURI = wrappedMessageType.getNamespaceURI();
                payLoadAsStreamReader = XMLInputFactory.newInstance().createXMLStreamReader(bais);
                payLoadAsStreamReader.next(); // skip the <?xml ...?> node.
                payLoadAsSource = new StAXSource(payLoadAsStreamReader, true);
                
                printPayLoad("Unwrapped message = " + baos.toString());
            } catch(Exception ex) {
                logger.log(Level.SEVERE, ex.getMessage(), ex);
            }
        }
        
        private boolean isSimpleType(Part part) {
            QName parttype = part.getTypeName();
            if (parttype != null) {
                String s = parttype.getNamespaceURI();
                if ( s != null && s.trim().equals("http://www.w3.org/2001/XMLSchema")) {
                    return true;
                }
            }
            return false;
        }
        
        private void writeJBIParts(XMLStreamReader reader,
                XMLStreamWriter writer,
                List<Part> parts) throws Exception {
            long startTime = System.currentTimeMillis();
            /**
             * The following loop is expected to iterate only once always,
             * because there could be only one <jbi:message> in the request.
             */
            while(reader.hasNext()) {
                if(reader.next() != XMLStreamReader.START_ELEMENT ||
                        !isJBINode(reader, WRAPPER_MESSAGE_LOCALNAME)) {
                    continue;
                }
                /**
                 * The following loop is expected to iterate as much as there are <jbi:part> elements.
                 */
                int k = 0;
                while(reader.hasNext()) {
                    if(reader.next() != XMLStreamReader.START_ELEMENT ||
                            !isJBINode(reader, WRAPPER_PART_LOCALNAME)) {
                        continue;
                    }
                    Part part = parts.get(k++);
                    //boolean simpleType = isSimpleType(part);
                    //if(simpleType) {
                    writer.writeStartElement(part.getName());
                    //}
                    
                    DOMUtil.UTIL.writeChildren(reader, writer);
                    
                    //if(simpleType) {
                    writer.writeEndElement();
                    //}
                }
            }
            long timeTaken = System.currentTimeMillis() - startTime;
            logger.log(Level.FINE, "TimeTaken to write JBI parts to payload = " + timeTaken);
        }
        
        private void writeJBIParts(Node wrappedDocument,
                XMLStreamWriter writer,
                List<Part> parts) throws Exception {
            NodeList jbiMessages = wrappedDocument.getChildNodes();
            /**
             * The following loop is expected to iterate only once always,
             * because there could be only one <jbi:message> in the request.
             */
            long startTime = System.currentTimeMillis();
            for(int i=0; i<jbiMessages.getLength(); i++) {
                Node jbiMessage = jbiMessages.item(i);
                if(!isJBINode(jbiMessage, WRAPPER_MESSAGE_LOCALNAME)) continue;
                NodeList jbiParts = jbiMessage.getChildNodes();
                /**
                 * The following loop is expected to iterate as much as
                 * there are <jbi:part> elements.
                 */
                for(int j = 0, k = 0; j< jbiParts.getLength(); j++) {
                    Node jbiPart = jbiParts.item(j);
                    if(!isJBINode(jbiPart, WRAPPER_PART_LOCALNAME)) continue;
                    
                    Part part = parts.get(k++);
                    //boolean simpleType = isSimpleType(part);
                    
                    //if(simpleType) {
                    writer.writeStartElement(part.getName());
                    //}
                    
                    DOMUtil.UTIL.writeChildren(writer, jbiPart);
                    
                    //if(simpleType) {
                    writer.writeEndElement();
                    //}
                }
            }
            long timeTaken = System.currentTimeMillis() - startTime;
            logger.log(Level.FINE, "TimeTaken to write JBI parts to payload = " + timeTaken);
        }
        
    }
    
    private boolean isJBINode(Node node, String localName) {
        if(WRAPPER_DEFAULT_NAMESPACE.equalsIgnoreCase(node.getNamespaceURI()) &&
                localName.equalsIgnoreCase(node.getLocalName())) {
            return true;
        }
        return false;
    }
    
    private boolean isJBINode(XMLStreamReader reader, String localName) {
        if(WRAPPER_DEFAULT_NAMESPACE.equalsIgnoreCase(reader.getNamespaceURI()) &&
                localName.equalsIgnoreCase(reader.getLocalName())) {
            return true;
        }
        return false;
    }
    
    public Node getOwnerDocument(Node n) {
        Node ownerDocument = n.getOwnerDocument();
        return ownerDocument != null
                ? ownerDocument
                : n;
    }
}
