/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.jbi.serviceengine.core;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import javax.xml.namespace.QName;

/**
 * Registry of all ServiceEndpoints, provides mapping between EndPointName and
 * ServiceEngineEndpoint.
 * @author Manisha Umbarje
 */
public class EndpointRegistry {
    
    /** Service QName to table of endpoints
     * The key being the service name and value being table of ServiceEngineEndpoints
     */
    private ConcurrentHashMap endpoints;
    private ConcurrentHashMap<String, DescriptorEndpointInfo> jbiProviders;
    private ConcurrentHashMap<String, DescriptorEndpointInfo> jbiConsumers;
    private Set<String> compApps;
    
    private static EndpointRegistry store = new EndpointRegistry();
    
    /** Creates a new instance of ServiceEndPointRegistry */
    private EndpointRegistry() {
        endpoints = new ConcurrentHashMap(11,0.75f,4);
        jbiProviders = new ConcurrentHashMap<String,DescriptorEndpointInfo>(11,0.75f,4);
        jbiConsumers = new ConcurrentHashMap<String,DescriptorEndpointInfo>(11,0.75f,4);
        compApps = new HashSet<String>();
    }
    
    public static EndpointRegistry getInstance() {
        return store;
    }
    
    
    
    /**
     * Adds a ServiceEndpoint to the store
     */
    public void put(QName service, String endpointName, ServiceEngineEndpoint endpoint) {
        Map map=  (Map)endpoints.get(service);
        if(map == null) {
            map = new Hashtable();
            endpoints.put(service, map);
        }
        map.put(endpointName, endpoint);
        
    }
    
    /**
     *
     */
    public ServiceEngineEndpoint get(QName service, String endpointName) {
        
        Map map=  (Map)endpoints.get(service);
        if(map != null)
        return (ServiceEngineEndpoint)map.get(endpointName);
        else
            return null;
        
    }
    
    /**
     * Removes ServiceEndpoint from the store
     */
    public void delete(QName service, String endpointName) {
        
        Map map=  (Map)endpoints.get(service);
        map.remove(endpointName);
        
    }
    
    public List<ServiceEngineEndpoint> list() {
        List<ServiceEngineEndpoint> list = new LinkedList<ServiceEngineEndpoint>();
        for (Iterator itr = endpoints.values().iterator();itr.hasNext();) {
            Hashtable table = (Hashtable) itr.next();
            list.addAll(table.values());
        }
        return list;
    }

    /**
     * Check whether this endpoint is provided by any composite application.
     */
    public boolean hasProviderEP(ServiceEngineEndpoint seEndpoint) {
        //if the javaee app is not deployed as part of comp app then return true
        if(!compApps.contains(seEndpoint.getApplicationName()))
            return true;
        
        QName service = seEndpoint.getServiceName();
        String endpointName = seEndpoint.getEndpointName();
        DescriptorEndpointInfo ep = 
                jbiProviders.get(service.getLocalPart() + endpointName);
        return ep !=null && ep.isStarted();
    }

    public boolean hasConsumerEP(QName service, String endpointName) {
        DescriptorEndpointInfo ep = 
                jbiConsumers.get(service.getLocalPart() + endpointName);
        return ep !=null && ep.isStarted();
    }

    /**
     * The APIs below are used by JBIEndpointManager.RegistryManager to populate
     * the entries
     */
    ConcurrentHashMap<String, DescriptorEndpointInfo> getProviders() {
        return jbiProviders;
    }
    
    ConcurrentHashMap<String, DescriptorEndpointInfo> getConsumers() {
        return jbiConsumers;
    }
    
    Set<String> getCompApps() {
        return compApps;
    }
}
