/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.jbi.serviceengine.core;

import java.io.InputStream;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.w3c.dom.Document;

import javax.xml.parsers.DocumentBuilderFactory;
import java.io.File;
import java.io.FileInputStream;
import java.util.List;
import java.util.LinkedList;
import java.util.logging.Logger;
import java.util.logging.Level;
import java.util.concurrent.ConcurrentHashMap;

import com.sun.enterprise.jbi.serviceengine.bridge.EndpointHelper;
import com.sun.enterprise.jbi.serviceengine.util.soap.StringTranslator;
import com.sun.logging.LogDomains;

/**
 * Class to manage enabling and disabling of endpoints in NMR. It uses 
 * EndpointHelper to manage the endpoints.
 * 
 * @author Vikas Awasthi
 */
public class JBIEndpointManager {

    private RegistryManager registryManager;
    private final Logger logger = LogDomains.getLogger(LogDomains.SERVER_LOGGER);
    private StringTranslator translator;
    
    JBIEndpointManager() {
        registryManager = new RegistryManager();
        translator = new StringTranslator(this.getClass().getPackage().getName(), 
                                        this.getClass().getClassLoader());
    }

    /**
     * Parse the jbi.xml file and read enpoint information. Store this 
     * information in the EndpointRegistry. JBIDescriptorReader is used to 
     * read jbi.xml. It creates a list of DescriptorEndpointInfo objects 
     * corresponding to the endpoints specified in the jbi.xml
     * 
     * If there is any exception while parsing jbi.xml then deployment of the 
     * service unit will be stopped.
     */
    void storeAllEndpoints(String appLocation, String su_Name) throws Exception {
        InputStream jbi_xml = null;
        try {
            if(new File(appLocation).isDirectory()) {
                jbi_xml = new FileInputStream(appLocation +
                        File.separator +
                        "META-INF" +
                        File.separator +
                        "jbi.xml");
            } else {
                ZipFile zipFile = new ZipFile(appLocation);
                for (Enumeration e = zipFile.entries(); e.hasMoreElements();) {
                    ZipEntry zipEntry = (ZipEntry) e.nextElement();
                    if("META-INF/jbi.xml".equals(zipEntry.getName())) {
                        jbi_xml = zipFile.getInputStream(zipEntry);
                        break;
                    }
                }
            }
 
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            factory.setValidating(false);
            Document doc = factory.newDocumentBuilder().parse(jbi_xml);
            JBIDescriptorReader reader = new JBIDescriptorReader(su_Name);
            reader.init(doc);
            registryManager.addAllEndpointInfo(reader.getEndpoints());
            registryManager.addAllProviderEPForApp(su_Name);
            registryManager.addServiceUnit(su_Name);
            
            EndpointHelper epHelper = EndpointHelper.getInstance();
            // Now disable all the endpoints for this application. This step is
            // required because by default all the endpoints in a webservice app
            // are enabled in NMR. If this default behavior is changed in future
            // then the code below can be removed.
            for (DescriptorEndpointInfo ep : 
                    registryManager.getAllProviderEPForApp(su_Name)) {
                epHelper.disableEndpoint(ep.getServiceName(),ep.getEndpointName());
            }
        } catch (Exception e) {
            String exceptionText = 
                    translator.getString("serviceengine.jbixml_readerror",su_Name);
            logger.log(Level.SEVERE,exceptionText,e);
            throw new Exception(exceptionText, e);
        } finally {
            if(jbi_xml!=null)
                jbi_xml.close();
        }
    }

    /**
     * Clear the registry. This will be called during service unit undeployment.
     */
    void removeAllEndpoints(String su_Name) {
        // Enable all the endpoints (only if jbi-enabled true)
        EndpointHelper epHelper = EndpointHelper.getInstance();
        for (DescriptorEndpointInfo ep : 
                registryManager.getAllProviderEPForApp(su_Name)) {
            epHelper.checkAndEnableEndpoint(ep.getServiceName(), 
                                            ep.getEndpointName());
        }

        registryManager.removeAllConsumerEP(su_Name);
        registryManager.removeAllProviderEP(su_Name);
        registryManager.removeAllProviderEPForApp(su_Name);
        registryManager.removeServiceUnit(su_Name);
    }

    /**
     * Start all the endpoints specified in jbi.xml. In case of consumer just 
     * the state variable is set to started. For providers all the endpoints are
     * activated in NMR.  
     */
    void startAllEndpoints(String su_Name) throws Exception {
        EndpointHelper epHelper = EndpointHelper.getInstance();
        for (DescriptorEndpointInfo ep : registryManager.getAllConsumerEP(su_Name))
            ep.setStarted(true);

        for (DescriptorEndpointInfo ep : registryManager.getAllProviderEP(su_Name)) {
            if(!registryManager.hasAppserverEndpoint(ep)) 
                throw new Exception(
                        translator.getString("serviceengine.endpoint_mismatch", 
                                            ep.getServiceName().getLocalPart(), 
                                            su_Name));

            ep.setStarted(true);
            // now activate the endpoint in NMR
            epHelper.enableEndpoint(ep.getServiceName(), ep.getEndpointName());
        }
    }

    /**
     * Disable the endpoints enabled in during start 
     */
    void stopAllEndpoints(String su_Name) {
        EndpointHelper epHelper = EndpointHelper.getInstance();
        for (DescriptorEndpointInfo ep : registryManager.getAllConsumerEP(su_Name))
            ep.setStarted(false);

        for (DescriptorEndpointInfo ep : registryManager.getAllProviderEP(su_Name)) {
            ep.setStarted(false);
            // now de-activate the endpoint in NMR
            epHelper.disableEndpoint(ep.getServiceName(), ep.getEndpointName());
        }
    }

    /** 
     * Inner class to manage jbi endpoints in the registry. As this class is 
     * used within JBIEndpoint manager this is made Inner rather than a 
     * separate class. 
     */
    class RegistryManager {

        RegistryManager() {
            provider_endpoints = epRegistry.getProviders();
            consumer_endpoints = epRegistry.getConsumers();
            provider_endpointsForApp =
                    new ConcurrentHashMap<String, DescriptorEndpointInfo>();
        }


        void addAllEndpointInfo(DescriptorEndpointInfo[] endpoints) {
            for (DescriptorEndpointInfo ep : endpoints) {
                if(ep.isProvider())
                    provider_endpoints.put(ep.getKey(), ep);
                else
                    consumer_endpoints.put(ep.getKey(), ep);
            }
        }
        
        void addServiceUnit(String su_Name) {
            epRegistry.getCompApps().add(su_Name);
        }

        void removeServiceUnit(String su_Name) {
            epRegistry.getCompApps().remove(su_Name);
        }

        /**
         * Add the endpoints of a webservice application not specified in 
         * jbi.xml. This list is used to disable all endpoints from NMR during
         * the service unit deployment step. 
         * The su_Name here should be the JavaEE application name.
         */
        void addAllProviderEPForApp(String su_Name) {
            for (ServiceEngineEndpoint seEndpoint : epRegistry.list()) {
                if (seEndpoint.getApplicationName().equals(su_Name)) {
                    DescriptorEndpointInfo ep =
                            new DescriptorEndpointInfo(su_Name);
                    ep.setEndpointName(seEndpoint.getEndpointName());
                    ep.setServiceName(seEndpoint.getServiceName());
                    ep.setProvider();
                    provider_endpointsForApp.put(ep.getKey(), ep);
                }
            }
        }

        void removeAllProviderEP(String su_Name) {
            for (String key : provider_endpoints.keySet()) {
                DescriptorEndpointInfo ep = provider_endpoints.get(key);
                if(ep!=null && ep.getSu_Name().equals(su_Name))
                    provider_endpoints.remove(key);
            }
        }

        void removeAllProviderEPForApp(String su_Name) {
            for (String key : provider_endpointsForApp.keySet()) {
                DescriptorEndpointInfo ep = provider_endpointsForApp.get(key);
                if(ep!=null && ep.getSu_Name().equals(su_Name))
                    provider_endpointsForApp.remove(key);
            }
        }

        void removeAllConsumerEP(String su_Name) {
            for (String key : consumer_endpoints.keySet()) {
                DescriptorEndpointInfo ep = consumer_endpoints.get(key);
                if(ep!=null && ep.getSu_Name().equals(su_Name))
                    consumer_endpoints.remove(key);
            }
        }

        List<DescriptorEndpointInfo> getAllProviderEP(String su_Name) {
            List<DescriptorEndpointInfo> list = 
                    new LinkedList<DescriptorEndpointInfo>();
            
            for (String key : provider_endpoints.keySet()) {
                DescriptorEndpointInfo ep = provider_endpoints.get(key);
                if(ep!=null && ep.getSu_Name().equals(su_Name))
                    list.add(ep);
            }
            return list;
        }

        List<DescriptorEndpointInfo> getAllConsumerEP(String su_Name) {
            List<DescriptorEndpointInfo> list = 
                    new LinkedList<DescriptorEndpointInfo>();
            
            for (String key : consumer_endpoints.keySet()) {
                DescriptorEndpointInfo ep = consumer_endpoints.get(key);
                if(ep!=null && ep.getSu_Name().equals(su_Name))
                    list.add(ep);
            }
            return list;
        }

        List<DescriptorEndpointInfo> getAllProviderEPForApp(String su_Name) {
            List<DescriptorEndpointInfo> list = 
                    new LinkedList<DescriptorEndpointInfo>();
            
            for (String key : provider_endpointsForApp.keySet()) {
                DescriptorEndpointInfo ep = provider_endpointsForApp.get(key);
                if(ep!=null && ep.getSu_Name().equals(su_Name))
                    list.add(ep);
            }
            return list;
        }

        /** Check whether appserver already has the given endpoint */
        boolean hasAppserverEndpoint(DescriptorEndpointInfo ep) {
            return (epRegistry.get(ep.getServiceName(),ep.getEndpointName())!=null);
        }

        // Using vectors here might be easier while updating but searching would
        // be costlier. Searching in HashMap is faster.
        // Since DescriptorEndpointInfo has serviceUnitName there is no need to
        // have a data structure that maintains endpointInfo Vs su_Name
        private ConcurrentHashMap<String, DescriptorEndpointInfo> provider_endpoints;
        private ConcurrentHashMap<String, DescriptorEndpointInfo> provider_endpointsForApp;
        private ConcurrentHashMap<String, DescriptorEndpointInfo> consumer_endpoints;
        private EndpointRegistry epRegistry = EndpointRegistry.getInstance();
    }
}
