/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment.annotation.handlers;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;

import com.sun.enterprise.deployment.EjbBundleDescriptor;
import com.sun.enterprise.deployment.EjbDescriptor;
import com.sun.enterprise.deployment.MethodDescriptor;

import com.sun.enterprise.deployment.annotation.AnnotatedElementHandler;
import com.sun.enterprise.deployment.annotation.AnnotationInfo;
import com.sun.enterprise.deployment.annotation.AnnotationProcessorException;
import com.sun.enterprise.deployment.annotation.HandlerProcessingResult;
import com.sun.enterprise.deployment.annotation.context.ComponentContext;
import com.sun.enterprise.deployment.annotation.context.EjbBundleContext;
import com.sun.enterprise.deployment.annotation.context.EjbContext;
import com.sun.enterprise.deployment.annotation.context.EjbsContext;
import com.sun.enterprise.deployment.annotation.context.EjbInterceptorContext;

import com.sun.enterprise.util.TypeUtil;

/**
 * This is an abstract class encapsulate generic behaviour of annotation
 * handler applying on Ejb Class.  It will get the corresponding
 * EjbDescriptors associated to the annotation on the given Ejb Class
 * and then pass it to underlying processAnnotation method.
 * Concrete subclass handlers need to implement the following:
 *     public Class&lt;? extends Annotation&gt; getAnnotationType();
 *     protected HandlerProcessingResult processAnnotation(AnnotationInfo ainfo,
 *             EjbContext[] ejbContexts) throws AnnotationProcessorException;
 * It may also need to override the following:
 * a) if other annotations need to be processed prior to given annotation:
 *     public Class&lt;? extends Annotation&gt;[] getTypeDependencies();
 * b) if the given annotation can be processed while processing another
 *    annotation
 *     protected boolean isDelegatee();
 * c) if we need to process for interceptor
 *     protected HandlerProcessingResult processAnnotation(AnnotationInfo ainfo,
 *             EjbInterceptorContext ejbInterceptorContext)
 *             throws AnnotationProcessorException;
 * d) indicate the annotation support type inheritance
 *     protected boolean supportTypeIneritance();
 *
 * @author Shing Wai Chan
 */
abstract class AbstractAttributeHandler extends AbstractHandler {
    /**
     * Process Annotation with given EjbContexts.
     * @param ainfo
     * @param ejbContexts
     * @return HandlerProcessingResult
     * @exception AnnotationProcessorException
     */
    protected abstract HandlerProcessingResult processAnnotation(
            AnnotationInfo ainfo, EjbContext[] ejbContexts)
            throws AnnotationProcessorException;

    /**
     * Process Annotation with given InteceptorContext.
     * @param ainfo
     * @param ejbInterceptorContext
     * @return HandlerProcessingResult
     * @exception AnnotationProcessorException
     */
    protected HandlerProcessingResult processAnnotation(
            AnnotationInfo ainfo, EjbInterceptorContext ejbInterceptorContext)
            throws AnnotationProcessorException {
        if (!isDelegatee()) {
            throw new UnsupportedOperationException();
        }
        return getDefaultProcessedResult();
    }

    /**
     * Process a particular annotation which type is the same as the
     * one returned by @see getAnnotationType(). All information
     * pertinent to the annotation and its context is encapsulated
     * in the passed AnnotationInfo instance.
     * This is a method in interface AnnotationHandler.
     *
     * @param ainfo the annotation information
     */
    public HandlerProcessingResult processAnnotation(AnnotationInfo ainfo) 
            throws AnnotationProcessorException {
        
        AnnotatedElement ae = ainfo.getAnnotatedElement();
        Annotation annotation = ainfo.getAnnotation();

        if (logger.isLoggable(Level.FINER)) {
            logger.finer("@process annotation " + annotation + " in " + ae);
        }

        AnnotatedElementHandler aeHandler = ainfo.getProcessingContext().getHandler();

        if (aeHandler instanceof EjbBundleContext) {
            EjbBundleContext ejbBundleContext = (EjbBundleContext)aeHandler;
            AnnotatedElementHandler aeh = ejbBundleContext.createContextForEjb();
            if (aeh != null) {
                aeHandler = aeh;
            } else {
                if (isDelegatee()) {
                    aeHandler = ejbBundleContext.createContextForEjbInterceptor();
                }
                if (aeHandler == null) {
                    return getInvalidAnnotatedElementHandlerResult(aeHandler, ainfo);
                }
            }
        }

        if (!supportTypeInheritance() &&
                ElementType.TYPE.equals(ainfo.getElementType()) &&
                aeHandler instanceof ComponentContext) {
            ComponentContext context = (ComponentContext)aeHandler;
            Class clazz = (Class)ainfo.getAnnotatedElement();
            if (!clazz.getName().equals(context.getComponentClassName())) {
                if (logger.isLoggable(Level.WARNING)) {
                    log(Level.WARNING, ainfo, 
                        localStrings.getLocalString(
                        "enterprise.deployment.annotation.handlers.typeinhernotsupp",
                        "The annotation symbol inheritance is not supported."));
                }
                return getDefaultProcessedResult();
            }
        }
                
        EjbContext[] ejbContexts = null;
        EjbInterceptorContext ejbInterceptorContext = null;
        if (aeHandler instanceof EjbContext) {
            EjbContext ejbContext = (EjbContext)aeHandler;
            ejbContexts = new EjbContext[] { ejbContext };
        } else if (aeHandler instanceof EjbsContext) {
            ejbContexts = ((EjbsContext)aeHandler).getEjbContexts();
        } else if (isDelegatee() && aeHandler instanceof EjbInterceptorContext) {
            ejbInterceptorContext = (EjbInterceptorContext)aeHandler;
        } else {
            return getInvalidAnnotatedElementHandlerResult(aeHandler, ainfo);
        }

        HandlerProcessingResult procResult = null;

        if (ejbInterceptorContext != null) {
            procResult = processAnnotation(ainfo, ejbInterceptorContext);
        } else {
            procResult = processAnnotation(ainfo, ejbContexts);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.finer("New annotation for " + annotation);
        }
        return procResult;
    }

    /**
     * This indicates whether the annotation can be processed by delegation
     * from the another annotation.
     */
    protected boolean isDelegatee() {
        return false;
    }

    /**
     * This indicates whether the annotation type should be processed for
     * type level in super-class.
     */
    protected boolean supportTypeInheritance() {
        return false;
    }

    /**
     * Returns MethodDescriptors representing All for a given EjbDescriptor.
     * @param ejbDesc
     * @return resulting MethodDescriptor
     */
    protected Set<MethodDescriptor> getMethodAllDescriptors(
            EjbDescriptor ejbDesc) {
        Set methodAlls = new HashSet();
        if (ejbDesc.isRemoteInterfacesSupported() ||
            ejbDesc.isRemoteBusinessInterfacesSupported()) {
            methodAlls.add(
                    new MethodDescriptor(MethodDescriptor.ALL_METHODS,
                    "", MethodDescriptor.EJB_REMOTE));
            if (ejbDesc.isRemoteInterfacesSupported()) {
                methodAlls.add(
                    new MethodDescriptor(MethodDescriptor.ALL_METHODS,
                    "", MethodDescriptor.EJB_HOME));
            }
        }

        if (ejbDesc.isLocalInterfacesSupported() ||
                ejbDesc.isLocalBusinessInterfacesSupported()) {
            methodAlls.add(
                    new MethodDescriptor(MethodDescriptor.ALL_METHODS,
                    "", MethodDescriptor.EJB_LOCAL));
            if (ejbDesc.isLocalInterfacesSupported()) {
                methodAlls.add(
                    new MethodDescriptor(MethodDescriptor.ALL_METHODS,
                    "", MethodDescriptor.EJB_LOCALHOME));
            }
        } 

        if (ejbDesc.hasWebServiceEndpointInterface()) {
            methodAlls.add(
                    new MethodDescriptor(MethodDescriptor.ALL_METHODS,
                    "", MethodDescriptor.EJB_WEB_SERVICE));
        }

        return methodAlls;
    }

    protected boolean hasMethodPermissionsFromDD(MethodDescriptor methodDesc,
            EjbDescriptor ejbDesc) {
        HashMap methodPermissionsFromDD = ejbDesc.getMethodPermissionsFromDD();
        if (methodPermissionsFromDD != null) {
            Set allMethods = ejbDesc.getMethodDescriptors();
            String ejbClassSymbol = methodDesc.getEjbClassSymbol();
            for (Object mdObjsObj : methodPermissionsFromDD.values()) {
                List mdObjs = (List)mdObjsObj;
                for (Object mdObj : mdObjs) {
                    MethodDescriptor md = (MethodDescriptor)mdObj;
                    for (Object style3MdObj :
                            md.doStyleConversion(ejbDesc, allMethods)) {
                        MethodDescriptor style3Md = (MethodDescriptor)style3MdObj;
                        if (methodDesc.equals(style3Md)) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }
}
