/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment.node.runtime;

import java.util.Map;
import java.util.Iterator;
import org.w3c.dom.Node;

import com.sun.enterprise.deployment.node.XMLElement;
import com.sun.enterprise.deployment.node.DeploymentDescriptorNode;
import com.sun.enterprise.deployment.node.runtime.web.WebBundleRuntimeNode;
import com.sun.enterprise.deployment.types.MessageDestinationReferenceContainer;
import com.sun.enterprise.deployment.MessageDestinationReferenceDescriptor;
import com.sun.enterprise.deployment.xml.RuntimeTagNames;

/**
 * This node is responsible for handling runtime descriptor
 * message-destination-ref tag
 *
 */
public class MessageDestinationRefNode extends DeploymentDescriptorNode {

    private MessageDestinationReferenceDescriptor descriptor;
    
       /**
    * @return the descriptor instance to associate with this XMLNode
    */    
    public Object getDescriptor() {
        return descriptor;
    }   
    
    /**
     * all sub-implementation of this class can use a dispatch table to map 
     * xml element to method name on the descriptor class for setting the 
     * element value. 
     *  
     * @return the map with the element name as a key, the setter method 
     * as a value
     */    
    protected Map getDispatchTable() {    
        Map table = super.getDispatchTable();
        table.put(RuntimeTagNames.JNDI_NAME, "setJndiName");
        return table;
    }
    
    /**
     * receives notiification of the value for a particular tag
     * 
     * @param element the xml element
     * @param value it's associated value
     */
    public void setElementValue(XMLElement element, String value) {
        if (RuntimeTagNames.MESSAGE_DESTINATION_REFERENCE_NAME.equals(
            element.getQName())) {
            Object parentNode = getParentNode();
            Object parentDesc = null;
            // in case of web
            if (parentNode instanceof WebBundleRuntimeNode) {
                parentDesc = ((WebBundleRuntimeNode) parentNode).getWebBundleDescriptor();
            // in case of appclient and ejb
            } else {
                parentDesc = getParentNode().getDescriptor();
            }

            if (parentDesc instanceof MessageDestinationReferenceContainer) {
                descriptor = ((MessageDestinationReferenceContainer) parentDesc).getMessageDestinationReferenceByName(value);
            }
        } else super.setElementValue(element, value);
    }
    
    /**
     * write the descriptor class to a DOM tree and return it
     *
     * @param parent node for the DOM tree
     * @param node name for the descriptor
     * @param the descriptor to write
     * @return the DOM tree top node
     */    
    public Node writeDescriptor(Node parent, String nodeName, 
        MessageDestinationReferenceDescriptor msgDestRef) {          
        Node msgDestRefNode = super.writeDescriptor(parent, nodeName, 
            msgDestRef);
        appendTextChild(msgDestRefNode, 
            RuntimeTagNames.MESSAGE_DESTINATION_REFERENCE_NAME, 
            msgDestRef.getName());
        appendTextChild(msgDestRefNode, RuntimeTagNames.JNDI_NAME, 
            msgDestRef.getJndiName());
        return msgDestRefNode;
    }  
    
    /**
     * writes all the runtime information for JMS destination references
     * 
     * @param parent node to add the runtime xml info
     * @param the J2EE component containing message destination references
     */        
    public static void writeMessageDestinationReferences(Node parent, 
        MessageDestinationReferenceContainer descriptor) {
        // message-destination-ref*
        Iterator msgDestRefs = 
            descriptor.getMessageDestinationReferenceDescriptors().iterator();
        if (msgDestRefs.hasNext()) {
            MessageDestinationRefNode messageDestinationRefNode = 
                new MessageDestinationRefNode();
            while (msgDestRefs.hasNext()) {
                messageDestinationRefNode.writeDescriptor(parent, 
                    RuntimeTagNames.MESSAGE_DESTINATION_REFERENCE, 
                    (MessageDestinationReferenceDescriptor) msgDestRefs.next());
            }
        }       
    }
}
