/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.security.store;

import com.sun.enterprise.util.SystemPropertyConstants;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import java.util.Enumeration;

/**
 * This class implements an adapter for password manipulation a JCEKS.
 * @author Shing Wai Chan
 */
public class PasswordAdapter {
    public static final String PASSWORD_ALIAS_KEYSTORE = "domain-passwords";
    private KeyStore _pwdStore = null;
    private String _keyFile = null;            
    private char[] _masterPassword = null;

    private char[] getMasterPassword() {
        return _masterPassword;
    }
    
    private void setMasterPassword(char[] smp) {
        _masterPassword = smp;
    }
    
    /**
     * Construct a PasswordAdapter with given Shared Master Password,
     * SMP using the default keyfile (domain-passwords.jceks)
     * @param smp master password
     * @throws CertificateException
     * @throws IOException
     * @throws KeyStoreException
     * @throws NoSuchAlgorithmException
     */    
    public PasswordAdapter(char[] smp)
            throws CertificateException, IOException,
            KeyStoreException, NoSuchAlgorithmException 
    {
        
        String keyfileName = System.getProperty(SystemPropertyConstants.INSTANCE_ROOT_PROPERTY) +
            File.separator + "config" + File.separator + PASSWORD_ALIAS_KEYSTORE;
        init(keyfileName, smp);
    }
    
    /**
     * Construct a PasswordAdapter with given Shared Master Password,
     * SMP.
     * @param keyfileName the jceks key file name
     * @param smp master password
     * @throws CertificateException
     * @throws IOException
     * @throws KeyStoreException
     * @throws NoSuchAlgorithmException
     */    
    public PasswordAdapter(String keyfileName, char[] smp)
            throws CertificateException, IOException,
            KeyStoreException, NoSuchAlgorithmException 
    {
        init(keyfileName, smp);
    }
    
    /**
     * Construct a PasswordAdapter with given Shared Master Password,
     * SMP.
     * @param keyfileName the jceks key file name
     * @param smp the master password
     * @exception CertificateException
     * @exception IOException
     * @exception KeyStoreException
     * @exception NoSuchAlgorithmException
     */
    private void init(String keyfileName, char[] smp)
            throws CertificateException, IOException,
            KeyStoreException, NoSuchAlgorithmException 
    {
        _keyFile = keyfileName;
        _pwdStore = KeyStore.getInstance("JCEKS");
        setMasterPassword(smp);
        BufferedInputStream bInput = null;
        File file = new File(keyfileName);
        if (file.exists()) {
            bInput = new BufferedInputStream(new FileInputStream(file));
        }
        try {
            //load must be called with null to initialize an empty keystore
            _pwdStore.load(bInput, getMasterPassword());
            if (bInput != null) {
                bInput.close();
                bInput = null;
            } 
        } finally {
             if (bInput != null) {
                 try {
                     bInput.close();
                 } catch(Exception ex) {
                     //ignore we are cleaning up
                 }
             }
        }        
    }
    
    /**
     * This methods returns password String for a given alias and SMP.
     * @param alias     
     * @return corresponding password or null if the alias does not exist.
     * @exception KeyStoreException
     * @exception NoSuchAlgorithmException
     * @exception UnrecoverableKeyException
     */
    public String getPasswordForAlias(String alias)
            throws KeyStoreException, NoSuchAlgorithmException,
            UnrecoverableKeyException {

         Key key = _pwdStore.getKey(alias, getMasterPassword()); 
         if (key != null) {
            return new String(key.getEncoded());
         } else {
             return null;
         }
    }

    /**
     * This methods returns password SecretKey for a given alias and SMP.
     * @param alias     
     * @return corresponding password SecretKey or
     *         null if the alias does not exist.
     * @exception KeyStoreException
     * @exception NoSuchAlgorithmException
     * @exception UnrecoverableKeyException
     */
    public SecretKey getPasswordSecretKeyForAlias(String alias)
            throws KeyStoreException, NoSuchAlgorithmException,
            UnrecoverableKeyException {

        return (SecretKey)_pwdStore.getKey(alias, getMasterPassword()); 
    }


    /**
     * See if the given alias exists
     * @param alias the alias name 
     * @return true if the alias exists in the keystore
     */    
    public boolean aliasExists(String alias) throws KeyStoreException
    {
        return _pwdStore.containsAlias(alias);
    }
    
    /**
     * Remove an alias from the keystore
     * @param alias The name of the alias to remove
     * @throws KeyStoreException
     * @throws IOException
     * @throws NoSuchAlgorithmException
     * @throws CertificateException
     */    
    public void removeAlias(String alias) throws KeyStoreException, IOException,
        NoSuchAlgorithmException, CertificateException
    {
        _pwdStore.deleteEntry(alias);
        writeStore();
    }    
    
    /**
     * Return the aliases from the keystore.
     * @return An enumeration containing all the aliases in the keystore.
     */    
    public Enumeration getAliases() throws KeyStoreException
    {        
        return _pwdStore.aliases();
    }
    
    /**
     * Writes the keystore to disk
     * @throws KeyStoreException
     * @throws IOException
     * @throws NoSuchAlgorithmException
     * @throws CertificateException
     */    
    public void writeStore() throws KeyStoreException, IOException, 
        NoSuchAlgorithmException, CertificateException
    {
         BufferedOutputStream boutput = null;

         try {
             boutput = new BufferedOutputStream(
                     new FileOutputStream(_keyFile));
             _pwdStore.store(boutput, getMasterPassword());
             boutput.close();
             boutput = null;
         } finally {
             if (boutput != null) {
                 try {
                     boutput.close();
                 } catch(Exception ex) {
                     //ignore we are cleaning up
                 }
             }
         }
    }
    
    /**
     * This methods set alias, secretKey into JCEKS keystore.
     * @param alias
     * @param secretKey     
     * @exception CertificateException
     * @exception IOException
     * @exception KeyStoreException
     * @exception NoSuchAlgorithmException
     */
    public void setPasswordForAlias(String alias, byte[] secretKey) 
        throws CertificateException, IOException, KeyStoreException, 
        NoSuchAlgorithmException 
    {
         Key key = new SecretKeySpec(secretKey, "AES");
         _pwdStore.setKeyEntry(alias, key, getMasterPassword(), null);
         writeStore();
    }
    
    /**
     * Changes the keystore password
     * @param newpassword the new keystore password
     * @throws KeyStoreException
     * @throws IOException
     * @throws NoSuchAlgorithmException
     * @throws CertificateException
     */    
    public void changePassword(char[] newpassword) throws KeyStoreException, IOException, 
        NoSuchAlgorithmException, CertificateException
    {
        setMasterPassword(newpassword);
        writeStore();
    }
}
