/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.util;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ExecutorService;

/**
    Base class (can be used directly) for running small, short lived tasks.  An ExecutorService
    is used for efficiency, and excess threads are discarded quickly.
    <p>
    Includes convenience routines for submitting tasks, determining result status.
    <p>
    <b>Example (inline usage)</b>
    <pre>
    final RunnableBase myTask = new RunnableBase( "Compute PI" ) {
        public void run() {
            final double pi = 3.141926386; // cheater method
        }
    };
    myTask.submit();
    ...
    // wait for compute task
    myTask.waitDoneThrow(); // or waitDone() followed by myTask.getThrowable()
    </pre>
 */
public abstract class  RunnableBase<T> implements Runnable
{
    /** a Throwable if anything was thrown from the run loop */
    private volatile Throwable      mThrowable;
    
    /** means to block client threads until done */
    private final CountDownLatch    mLatch;
    
    /** optional name of the task */
    private final String            mName;
    
    /** debugging: whether to sleep a random amount.  See {@link #setUseRandomSleep} */
    private volatile boolean        mUseRandomSleep;
    
    // optional data for use by the task
    private final T                 mData;
    
    private volatile long           mSubmitNanos;
    private volatile long           mRunStartNanos;
    private volatile long           mRunDoneNanos;
    
    // this should use AMXDebugHelper instead, but build order precludes it
    private static final boolean DEBUG_ENABLED = false;
        private void
    debug( final Object... args)
    {
        if ( DEBUG_ENABLED )
        {
            String msg  = "";
            for( int i = 0; i < args.length - 1; ++i )
            {
                msg = msg + args[i] + ", ";
            }
            msg = msg + args[args.length-1];
            
            System.out.println( msg );
        }
    }
    
    private static final ExecutorService   _Exec = Executors.newCachedThreadPool();
        private static void
    _submit( final RunnableBase r )
    {
        r.mSubmitNanos    = System.nanoTime();
        _Exec.submit( r );
    }
    
    /**
        Submit the task for execution.  The task might finish before this method returns or
        afterwards. See {@link #waitDone} and {@link #waitDoneThrow}.
     */
        public void
    submit( )
    {
        _submit( this );
    }
    
    /**
        Create a new task.
        @param name use-readable name of the task
        @param data optional arbitrary data (see {@link #getData})
     */
        protected
    RunnableBase( final String name, final T data )
    {
        mName       = name == null ? "<no_name_specified>" : name ;
        mData       = data;
        mThrowable  = null;
        mUseRandomSleep = true;
        mSubmitNanos    = 0;
        mRunStartNanos    = 0;
        mRunDoneNanos    = 0;
        
        mLatch   = new CountDownLatch(1);
    }
    
        protected
    RunnableBase( final T data )
    {
        this( null, data );
    }
    
    public final T    getData()   { return mData; }
    
        protected
    RunnableBase(  )
    {
        this( null );
    }
    
    /* subclass must implement doRun() */
    protected abstract void doRun() throws Exception;
    
        protected  static void
    sleepMillis( final long millis )
    {
        try
        {
            Thread.sleep( millis );
        }
        catch( InterruptedException e )
        {
        }
    } 
    
    private static final long MAX_RANDOM_SLEEP_MILLIS   = 500;
    
    /**
        Good for debugging timing issues; a task will insert an artificial delay
        by a random amount.
     */
        public void 
    setUseRandomSleep( final boolean useRandom )
    {
        mUseRandomSleep = useRandom;
    }
    
    /**
        May be called synchronously or via another thread {@link #submit}.
        See {@link #waitDone} and {@link #waitDoneThrow} and {@link #getThrowable}.
     */
        public final void
    run()
    {
        mRunStartNanos    = System.nanoTime();
        try
        {
            if ( mUseRandomSleep )
            {
                sleepMillis( mRunStartNanos % MAX_RANDOM_SLEEP_MILLIS );
            }
            
            doRun();
        }
        catch( Throwable t )
        {
            mThrowable  = t;
        }
        finally
        {
            mRunDoneNanos    = System.nanoTime();
            debug( toString() );
            mLatch.countDown();
        }
    }
    
    /**
        @return the number of nanoseconds to execute the task from the time it was submitted
     */
        public long
    getNanosFromSubmit()
    {
        return mRunDoneNanos - mSubmitNanos;
    }
    
    /**
        @return the number of nanoseconds to execute the task from the time it actually entered
        the {@link #run} method.
     */
        public long
    getNanosFromRunStart()
    {
        return mRunDoneNanos - mRunStartNanos;
    }
    
    /**
        @return the number of nanoseconds between task-submittal and the actual execution start
     */
        public long
    getRunLatency()
    {
        return mRunStartNanos - mSubmitNanos;
    }
    
    /**
        Block until the task has finished, and return any Throwable (hopefully null).
        @return the Throwable that was thrown (if any), otherwise null
     */
        public final Throwable
    waitDone()
    {
        try
        {
            mLatch.await();
        }
        catch( final InterruptedException intr )
        {
           throw new RuntimeException( intr );
        }
        
        return mThrowable;
    }
    
    /**
        Block until the task has finished.  If a Throwable was thrown, then this method
        will rethrow it, or a RuntimeException.
     */
        public final void
    waitDoneThrow()
    {
        final Throwable t   = waitDone();
        if ( t != null )
        {
            if ( t instanceof RuntimeException )
            {
                throw (RuntimeException)t;
            }
            else if ( t instanceof Error )
            {
                throw (Error)t;
            }
            else
            {
                throw new RuntimeException( t );
            }
        }
    }
    
        public String
    toString()
    {
        final String delim = ", ";
        
        final boolean started   = mSubmitNanos != 0;
        final boolean done      = mRunDoneNanos != 0;
        final long runTime      = started ?
            (done ? (mRunDoneNanos - mRunStartNanos) : System.nanoTime() - mRunStartNanos) : 0;
        final String throwable = mThrowable == null ? "" : mThrowable.toString();
        
        return "Runnable \"" + this.getClass().getName() + "\"" + delim + "name = " + mName +
            delim + "started=" + started + delim + "done=" + done +
            delim + "run-time=" + runTime + delim + throwable;
    }
};
































