/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.appclient.jws;

import com.sun.enterprise.security.SSLUtils;
import com.sun.enterprise.security.SecurityUtil;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.logging.LogDomains;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.Map;
import java.util.Vector;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import sun.misc.BASE64Encoder;
import sun.security.util.ManifestDigester;

/**
 * Signs a specified JAR file.
 *<p>
 *This implementation searches the available keystores for the signing alias
 *indicated in the domain.xml confirg or, if not specified, the default alias,
 *the first time it is invoked to sign a JAR file.  After the first requested
 *signing it uses the same alias and provider to sign all JARs.
 *<p>
 *The public interface to this class is the static signJar method.  
 *
 * @author tjquinn
 */
public class ASJarSigner {
    
    private static final String META_INF = "META-INF/";

    // prefix for new signature-related files in META-INF directory
    private static final String SIG_PREFIX = META_INF + "SIG-";

    /** property name optionally set by the admin in domain.xml to select an alias for signing */
    private static final String USER_SPECIFIED_ALIAS_PROPERTYNAME = "com.sun.aas.jws.signing.alias";

    /** default alias for signing if the admin does not specify one */
    private static final String DEFAULT_ALIAS_VALUE = "s1as";

    /** user-specified signing alias */
    private static final String userAlias = System.getProperty(USER_SPECIFIED_ALIAS_PROPERTYNAME);
    
    private static final Logger logger = 
            LogDomains.getLogger(LogDomains.CORE_LOGGER);

    private static final StringManager localStrings = StringManager.getManager(ASJarSigner.class);
    
    /** info used for signing, saved after being looked up during the first request */
    private static SigningInfo signingInfo = null;

    private File unsignedJar;
    private File signedJar;
    
    private ByteArrayOutputStream baos = new ByteArrayOutputStream(2048);
    private byte[] buffer = new byte[8192];

    /**
     *Creates a signed jar from the specified unsigned jar.
     *@param unsignedJar the unsigned JAR file
     *@param signedJar the signed JAR to be created
     *@return the elapsed time to sign the JAR (in milliseconds)
     *@throws Exception getting the keystores from SSLUtils fails
     */
    public static long signJar(File unsignedJar, File signedJar) throws Exception {
        synchronized(ASJarSigner.class) {
            if (signingInfo == null) {
                signingInfo = createSigningInfo();
            }
        }
        ASJarSigner signer = new ASJarSigner(unsignedJar, signedJar);
        return signer.sign();
    }
    
    /**
     * Creates a new instance of ASJarSigner
     */
    private ASJarSigner(File unsignedJar, File signedJar) throws Exception {
        this.unsignedJar = unsignedJar;
        this.signedJar = signedJar;
    }
    
    /**
     *Signs the input jar, creating the output jar.
     *<p>
     *Much of this method's implementation is from the JDK's JarSigner class which
     *implements the jarsigner command.  Large sections that do not apply to 
     *this usage have beem removed as compared to the original JarSigner code.
     */
    private long sign() throws ZipException, IOException, NoSuchAlgorithmException, ASJarSignerException {
        long startTime = System.currentTimeMillis();
        ZipFile zipFile = new ZipFile(unsignedJar);
        String sigfile = computeSigFileName(signingInfo.getAlias());
       
        FileOutputStream fos = new FileOutputStream(signedJar);
        
	PrintStream ps = new PrintStream(fos);
	ZipOutputStream zos = new ZipOutputStream(ps);

	/* First guess at what they might be - we don't xclude RSA ones. */
	String sfFilename = (META_INF + sigfile + ".SF").toUpperCase();
	String bkFilename = (META_INF + sigfile + ".DSA").toUpperCase();

	Manifest manifest = new Manifest();
	Map mfEntries = manifest.getEntries();

	boolean mfModified = false;
	boolean mfCreated = false;
	byte[] mfRawBytes = null;

	try {
	    // For now, hard-code the message digest algorithm to SHA-1
	    MessageDigest digests[] = { MessageDigest.getInstance("SHA1") };

	    // Check if manifest exists
	    ZipEntry mfFile;
	    if ((mfFile = getManifestFile(zipFile)) != null) {
		// Manifest exists. Read its raw bytes.
		mfRawBytes = getBytes(zipFile, mfFile);
		manifest.read(new ByteArrayInputStream(mfRawBytes));
	    } else {
		// Create new manifest
		Attributes mattr = manifest.getMainAttributes();
		mattr.putValue(Attributes.Name.MANIFEST_VERSION.toString(),
			       "1.0");
		String javaVendor = System.getProperty("java.vendor");
		String jdkVersion = System.getProperty("java.version");
		mattr.putValue("Created-By", jdkVersion + " (" +javaVendor
			       + ")");
		mfFile = new ZipEntry(JarFile.MANIFEST_NAME);
		mfCreated = true;
	    }

	    /*
	     * For each entry in jar
	     * (except for signature-related META-INF entries),
	     * do the following:
	     *
	     * - if entry is not contained in manifest, add it to manifest;
	     * - if entry is contained in manifest, calculate its hash and
	     *   compare it with the one in the manifest; if they are
	     *   different, replace the hash in the manifest with the newly
	     *   generated one. (This may invalidate existing signatures!)
	     */
	    BASE64Encoder encoder = new BASE64Encoder();
	    Vector mfFiles = new Vector();

	    for (Enumeration enum_=zipFile.entries();enum_.hasMoreElements();) {
		ZipEntry ze = (ZipEntry)enum_.nextElement();

		if (ze.getName().startsWith(META_INF)) {
		    // Store META-INF files in vector, so they can be written
		    // out first
		    mfFiles.addElement(ze);

		    if (signatureRelated(ze.getName())) {
			// ignore signature-related and manifest files
			continue;
		    }
		}

		if (manifest.getAttributes(ze.getName()) != null) {
		    // jar entry is contained in manifest, check and
		    // possibly update its digest attributes
		    if (updateDigests(ze, zipFile, digests, encoder,
				      manifest) == true) {
			mfModified = true;
		    }
		} else if (!ze.isDirectory()) {
		    // Add entry to manifest
		    Attributes attrs = getDigestAttributes(ze, zipFile,
							   digests,
							   encoder);
		    mfEntries.put(ze.getName(), attrs);
		    mfModified = true;
		}
	    }

	    // Recalculate the manifest raw bytes if necessary
	    if (mfModified) {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		manifest.write(baos);
		mfRawBytes = baos.toByteArray();
	    }

	    // Write out the manifest
	    if (mfModified) {
		// manifest file has new length
		mfFile = new ZipEntry(JarFile.MANIFEST_NAME);
	    }
	    zos.putNextEntry(mfFile);
	    zos.write(mfRawBytes);

	    // Calculate SignatureFile (".SF") and SignatureBlockFile
	    ManifestDigester manDig = new ManifestDigester(mfRawBytes);
	    SignatureFile sf = new SignatureFile(digests, manifest, manDig,
						 sigfile, true /*signManifest*/);

	    SignatureFile.Block block = null;

            block = sf.generateBlock(
                    signingInfo.getKey(), 
                    signingInfo.getCertificateChain(), 
                    true /*externalSF*/,
                    zipFile);

	    sfFilename = sf.getMetaName();
	    bkFilename = block.getMetaName();

	    ZipEntry sfFile = new ZipEntry(sfFilename);
	    ZipEntry bkFile = new ZipEntry(bkFilename);

	    long time = System.currentTimeMillis();
	    sfFile.setTime(time);
	    bkFile.setTime(time);

	    // signature file
	    zos.putNextEntry(sfFile);
	    sf.write(zos);

	    // signature block file
	    zos.putNextEntry(bkFile);
	    block.write(zos);

            // Write out all other META-INF files that we stored in the
	    // vector
	    for (int i=0; i<mfFiles.size(); i++) {
		ZipEntry ze = (ZipEntry)mfFiles.elementAt(i);
		if (!ze.getName().equalsIgnoreCase(JarFile.MANIFEST_NAME)
		    && !ze.getName().equalsIgnoreCase(sfFilename)
		    && !ze.getName().equalsIgnoreCase(bkFilename)) {
		    writeEntry(zipFile, zos, ze);
		}
	    }

	    // Write out all other files
	    for (Enumeration enum_=zipFile.entries();enum_.hasMoreElements();) {
		ZipEntry ze = (ZipEntry)enum_.nextElement();

		if (!ze.getName().startsWith(META_INF)) {
		    writeEntry(zipFile, zos, ze);
		}
	    }

            return System.currentTimeMillis() - startTime;
        } catch (Throwable t) {
            /*
             *In case of any problems, make sure there is no ill-formed signed
             *jar file left behind.
             */
            if (zos != null) {
                zos.close();
                zos = null;
            }
            signedJar.delete();

            /*
             *The jar signer will have written some information to System.out
             *and/or System.err.  Refer the user to those earlier messages.
             */
             throw new ASJarSignerException(localStrings.getString("jws.sign.errorSigning", signedJar.getAbsolutePath()), t);
        }
        finally {
            if (zos != null) {
                zos.close();
            }
            zipFile.close();
        }        
    }
    
    /*
     * Returns manifest entry from given jar file, or null if given jar file
     * does not have a manifest entry.
     */
    private ZipEntry getManifestFile(ZipFile zf) {
	ZipEntry ze = zf.getEntry(JarFile.MANIFEST_NAME);
	if (ze == null) {
	    // Check all entries for matching name
	    Enumeration enum_ = zf.entries();
	    while (enum_.hasMoreElements() && ze == null) {
		ze = (ZipEntry)enum_.nextElement();
		if (!JarFile.MANIFEST_NAME.equalsIgnoreCase
		    (ze.getName())) {
		    ze = null;
		}
	    }
	}
	return ze;
    }

    /**
     * signature-related files include:
     * . META-INF/MANIFEST.MF
     * . META-INF/SIG-*
     * . META-INF/*.SF
     * . META-INF/*.DSA
     * . META-INF/*.RSA
     */
    private boolean signatureRelated(String name) {
	String ucName = name.toUpperCase();
	if (ucName.equals(JarFile.MANIFEST_NAME) ||
	    ucName.equals(META_INF) ||
	    (ucName.startsWith(SIG_PREFIX) &&
		ucName.indexOf("/") == ucName.lastIndexOf("/"))) {
	    return true;
	}

	if (ucName.startsWith(META_INF) &&
	    isBlockOrSF(ucName)) {
	    // .SF/.DSA/.RSA files in META-INF subdirs
	    // are not considered signature-related
	    return (ucName.indexOf("/") == ucName.lastIndexOf("/"));
	}

	return false;
    }
    
    /**
     * From SignatureFileVerifier.
     * Utility method used by JarVerifier and JarSigner
     * to determine the signature file names and PKCS7 block
     * files names that are supported
     *
     * @param s file name
     * @return true if the input file name is a supported
     *          Signature File or PKCS7 block file name
     */
    private static boolean isBlockOrSF(String s) {
	// we currently only support DSA and RSA PKCS7 blocks
	if (s.endsWith(".SF") || s.endsWith(".DSA") || s.endsWith(".RSA")) {
	    return true;
	}
	return false;
    }

    /**
     *Derives the signature file name based on the alias.
     *<p>
     *Much of this logic is copied from the signJar method in the JDK's
     *tools.jar JarSigner class.
     *
     *@param alias the alias to be used in signing the JAR
     *@return the signature file name
     */
    private String computeSigFileName(String alias) {
        String sigfile = alias;
        boolean aliasUsed = true;
        
	if (sigfile.length() > 8) {
	    sigfile = sigfile.substring(0, 8).toUpperCase();
	} else {
	    sigfile = sigfile.toUpperCase();
	}

	StringBuffer tmpSigFile = new StringBuffer(sigfile.length());
	for (int j = 0; j < sigfile.length(); j++) {
	    char c = sigfile.charAt(j);
	    if (!
		((c>= 'A' && c<= 'Z') ||
		(c>= '0' && c<= '9') ||
		(c == '-') ||
		(c == '_'))) {
	    }
	    tmpSigFile.append(c);
	}

	sigfile = tmpSigFile.toString();

        return sigfile;
    }
    
    /*
     * Computes the digests of a zip entry, and returns them as a list of
     * attributes
     */
    private Attributes getDigestAttributes(ZipEntry ze, ZipFile zf,
					   MessageDigest[] digests,
					   BASE64Encoder encoder)
	throws IOException {

	String[] base64Digests = getDigests(ze, zf, digests, encoder);
	Attributes attrs = new Attributes();

	for (int i=0; i<digests.length; i++) {
	    attrs.putValue(digests[i].getAlgorithm()+"-Digest",
			   base64Digests[i]);
	}
	return attrs;
    }

    /*
     * Updates the digest attributes of a manifest entry, by adding or
     * replacing digest values.
     * A digest value is added if the manifest entry does not contain a digest
     * for that particular algorithm.
     * A digest value is replaced if it is obsolete.
     *
     * Returns true if the manifest entry has been changed, and false
     * otherwise.
     */
    private boolean updateDigests(ZipEntry ze, ZipFile zf,
				  MessageDigest[] digests,
				  BASE64Encoder encoder,
				  Manifest mf) throws IOException {
	boolean	update = false;

	Attributes attrs = mf.getAttributes(ze.getName());
	String[] base64Digests = getDigests(ze, zf, digests, encoder);

	for (int i=0; i<digests.length; i++) {
	    String name = digests[i].getAlgorithm()+"-Digest";
	    String mfDigest = attrs.getValue(name);
	    if (mfDigest == null
		&& digests[i].getAlgorithm().equalsIgnoreCase("SHA")) {
		// treat "SHA" and "SHA1" the same
		mfDigest = attrs.getValue("SHA-Digest");
	    }
	    if (mfDigest == null) {
		// compute digest and add it to list of attributes
		attrs.putValue(name, base64Digests[i]);
		update=true;
	    } else {
		// compare digests, and replace the one in the manifest
		// if they are different
		if (!mfDigest.equalsIgnoreCase(base64Digests[i])) {
		    attrs.putValue(name, base64Digests[i]);
		    update=true;
		}
	    }
	}
	return update;
    }

    /*
     * Computes the digests of a zip entry, and returns them as an array
     * of base64-encoded strings.
     */
    private synchronized String[] getDigests(ZipEntry ze, ZipFile zf,
					     MessageDigest[] digests,
					     BASE64Encoder encoder)
	throws IOException {

	int n, i;
	InputStream is = zf.getInputStream(ze);
	long left = ze.getSize();
	while((left > 0)
	      && (n = is.read(buffer, 0, buffer.length)) != -1) {
	    for (i=0; i<digests.length; i++) {
		digests[i].update(buffer, 0, n);
	    }
	    left -= n;
	}
	is.close();

	// complete the digests
	String[] base64Digests = new String[digests.length];
	for (i=0; i<digests.length; i++) {
	    base64Digests[i] = encoder.encode(digests[i].digest());
	}
	return base64Digests;
    }

    private void writeEntry(ZipFile zf, ZipOutputStream os, ZipEntry ze)
    throws IOException
    {
	byte[] data = getBytes(zf, ze);
        ZipEntry ze2 = new ZipEntry(ze.getName());
        ze2.setMethod(ze.getMethod());
        ze2.setTime(ze.getTime());
        ze2.setComment(ze.getComment());
        ze2.setExtra(ze.getExtra());
        if (ze.getMethod() == ZipEntry.STORED) {
            ze2.setSize(ze.getSize());
            ze2.setCrc(ze.getCrc());
        }
	os.putNextEntry(ze2);
	if (data.length > 0) {
	    os.write(data);
	}
    }

    private synchronized byte[] getBytes(ZipFile zf,
					 ZipEntry ze) throws IOException {
	int n;

	InputStream is = zf.getInputStream(ze);
	baos.reset();
	long left = ze.getSize();

	while((left > 0) && (n = is.read(buffer, 0, buffer.length)) != -1) {
	    baos.write(buffer, 0, n);
	    left -= n;
	}

	is.close();

	return baos.toByteArray();
    }

    /**
     *Wraps any underlying exception.
     *<p>
     *This is primarily used to insulate calling logic from
     *the large variety of exceptions that can occur during signing
     *from which the caller cannot really recover.
     */
    public class ASJarSignerException extends Exception {
        public ASJarSignerException(String msg, Throwable t) {
            super(msg, t);
        }
    }
    
    /**
     *Returns the signing info to use, creating it if it is not already
     *created.
     *@return the signing information to use in signing JARs
     */
    private static synchronized SigningInfo getSigningInfo() throws Exception {
        if (signingInfo == null) {
            signingInfo = createSigningInfo();
        }
        return signingInfo;
    }
    
    /**
     *Creates an object containing the alias and the keystore in which that
     *alias was found.
     *@return the SigningInfo object containing the information used during signing
     */
    private static SigningInfo createSigningInfo() throws Exception {
        SigningInfo defaultAliasSigningInfo = null;

        /*
         *Iterate through the available keystores.  If the user specified an
         *alias then exit as soon as we find a keystore containing it.  Along
         *the way remember the first occurrence of the default alias.  If
         *the user specified an alias but we never find it then we use the
         *default alias.
         */
        String[] keystorePWs = SecurityUtil.getSecuritySupport().getKeyStorePasswords();
        int keystoreSlot = 0;
        
        for (KeyStore ks : SSLUtils.getKeyStores()) {
            if (userAlias != null && ks.containsAlias(userAlias)) {
                /*
                 *The user specified an alias and the current keystore contains
                 *it.  Use this keystore and the user's alias.
                 */
                return new SigningInfo(userAlias, keystorePWs[keystoreSlot], ks);
            } else if (defaultAliasSigningInfo == null && 
                        ks.containsAlias(DEFAULT_ALIAS_VALUE)) {
                /*
                 *This is the first keystore that contains the default alias.
                 *Record this now in case we need it later.
                 */
                defaultAliasSigningInfo = new SigningInfo(DEFAULT_ALIAS_VALUE, 
                        keystorePWs[keystoreSlot], ks);
                /*
                 *If the user has not specified an alias then as soon as we
                 *find a keystore containing the default alias we can return.
                 */
                if (userAlias == null) {
                    return defaultAliasSigningInfo;
                }
            }
            keystoreSlot++;
        }

        /*
         *If the previous loop exits then the user specified an alias that
         *we could not find.  Warn that we will try to use the default alias instead.
         */
        logger.log(Level.WARNING, localStrings.getString("jws.sign.userAliasAbsent", 
                                                        userAlias));

        if (defaultAliasSigningInfo == null) {
            throw new IllegalArgumentException(
                    localStrings.getString("jws.sign.defaultAliasAbsent", 
                                            DEFAULT_ALIAS_VALUE));
        }
        return defaultAliasSigningInfo;
    }
    
    /**
     *Represents the information needed to actually sign a JAR file: the alias
     *and the keystore that contains the certificates associated with that alias.
     */
    private static class SigningInfo {
        private KeyStore keystore;
        private String alias;
        private String password;
        private PrivateKey key;
        
        public SigningInfo(String alias, String password, KeyStore keystore) throws KeyStoreException, NoSuchAlgorithmException, UnrecoverableKeyException {
            this.keystore = keystore;
            this.alias = alias;
            this.password = password;
            key = validateKey();
        }
        
        public String getAlias() {
            return alias;
        }
        
        private PrivateKey validateKey() throws KeyStoreException, NoSuchAlgorithmException, 
                UnrecoverableKeyException {
            Key tempKey = keystore.getKey(alias, password.toCharArray());
            if (tempKey instanceof PrivateKey) {
                return (PrivateKey) tempKey;
            } else {
                throw new IllegalArgumentException(localStrings.getString("jws.sign.keyNotPrivate", alias));
            }
        }
        
        public String getProviderName() {
            return keystore.getProvider().getName();
        }
        
        public String getPassword() {
            return password;
        }
        
        public String getStoreType() {
            return keystore.getType();
        }
        public X509Certificate[] getCertificateChain() throws KeyStoreException {
            Certificate[] certs = keystore.getCertificateChain(alias);
            X509Certificate[] X509certs = new X509Certificate[certs.length];
            int slot = 0;
            for (Certificate c : certs) {
                if (c instanceof X509Certificate) {
                    X509certs[slot++] = (X509Certificate) c;
                } else {
                    throw new IllegalArgumentException(localStrings.getString("jws.sign.notX509Cert", alias));
                }
            }
            return X509certs;
        }
        
        public String toString() {
            return new StringBuilder().append("Alias ").append(alias).
                    append("; ").append(keystore.toString()).toString();
        }
        
        public PrivateKey getKey() {
            return key;
        }
    }
    
    private static Constructor findConstructor(Class c, Class... argTypes) throws NoSuchMethodException {
        Constructor ct = c.getDeclaredConstructor(argTypes);
        if (ct == null) {
            throw new RuntimeException(localStrings.getString("jws.sign.noconst", c.getName()));
        }
        ct.setAccessible(true);
        return ct;
    }
    
    private static Method findMethod(Class c, String methodName, Class... argTypes) throws NoSuchMethodException {
        Method m = c.getDeclaredMethod(methodName, argTypes);
        if (m == null) {
            throw new RuntimeException(localStrings.getString(
                       "jws.sign.nomethod", methodName, c.getName()));
        }
        m.setAccessible(true);
        return m;
    }
    
    private static Class findInnerClass(Class c, String innerClassName) throws ClassNotFoundException {
        Class[] innerClasses = c.getDeclaredClasses();
        for (Class ic : innerClasses) {
            if (ic.getName().equals(innerClassName)) {
                return ic;
            }
        }
        throw new ClassNotFoundException(
                localStrings.getString("jws.sign.noinnerclass", innerClassName, c.getName()));
    }
    
    /**
     *Inspired by the JDK's SignatureFile class.
     *<p>
     *This class wraps the JDK's actual SignatureFile class, using reflection
     *to instantiate it and invokes its methods.
     */
    private class SignatureFile {

        private Object sigFile;
        
        private Class JDKsfClass;
        
        private Method getMetaNameMethod;
        private Method writeMethod;
        
        private static final String JDK_SIGNATURE_FILE = "sun.security.tools.SignatureFile";
        private static final String GETMETANAME_METHOD = "getMetaName";
        private static final String WRITE_METHOD = "write";
        
        public SignatureFile(MessageDigest digests[],
                             Manifest mf,
                             ManifestDigester md,
                             String baseName,
                             boolean signManifest) throws ClassNotFoundException, NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException {
       
           JDKsfClass = getClass().forName(JDK_SIGNATURE_FILE);
           
           /*
            *Locate the constructor and create an instance of the JDK's
            *SignatureFile class.
            */
           Constructor constructor = findConstructor(JDKsfClass,
                   MessageDigest[].class, 
                   Manifest.class,
                   ManifestDigester.class,
                   String.class,
                   Boolean.TYPE);
           
           sigFile = constructor.newInstance(digests, mf, md, baseName, signManifest);
           
           getMetaNameMethod = findMethod(JDKsfClass, GETMETANAME_METHOD);
           writeMethod = findMethod(JDKsfClass, WRITE_METHOD, OutputStream.class);
        }
        
        public Block generateBlock(PrivateKey privateKey,
                                   X509Certificate[] certChain,
                                   boolean externalSF, /* String tsaUrl,
                                   X509Certificate tsaCert,
                                   ContentSigner signingMechanism,
                                   String[] args, */ ZipFile zipFile)
            throws NoSuchAlgorithmException, InvalidKeyException, IOException,
                SignatureException, CertificateException, ClassNotFoundException, NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException
        {
            return new Block(this, privateKey, certChain, externalSF, /* tsaUrl,
                tsaCert, signingMechanism, args, */ zipFile);
        }
        
        public Class getJDKSignatureFileClass() {
            return JDKsfClass;
        }
        
        public Object getJDKSignatureFile() {
            return sigFile;
        }
        
        public String getMetaName() throws IllegalAccessException, InvocationTargetException {
            return (String) getMetaNameMethod.invoke(sigFile);
        }
        
        public void write(OutputStream os) throws IllegalAccessException, InvocationTargetException {
            writeMethod.invoke(sigFile, os);
        }

        /**
         *Inspired by the JDK's SignatureFile.Block inner class.
         *<p>
         *This inner class wraps the JDK's SignatureFile.Block inner class.
         */
        private class Block {

            private Object block;

            private static final String JDK_BLOCK = JDK_SIGNATURE_FILE + "$Block";
            private static final String JDK_CONTENT_SIGNER = "com.sun.jarsigner.ContentSigner";
            
            private Method getMetaNameMethod;
            private Method writeMethod;

            public Block(SignatureFile sfg, PrivateKey privateKey, 
                X509Certificate[] certChain, boolean externalSF, /* String tsaUrl,
                X509Certificate tsaCert, ContentSigner signingMechanism,
                String[] args, */ ZipFile zipFile) throws ClassNotFoundException, NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException {

                Class blockClass = getClass().forName(JDK_BLOCK);

                Class contentSignerClass = getClass().forName(JDK_CONTENT_SIGNER);

                /*
                 *Locate the constructor and create an instance of the JDK's
                 *Block class.
                 */
                Constructor constructor = findConstructor(blockClass,
                           sfg.getJDKSignatureFileClass(),
                           PrivateKey.class,
                           X509Certificate[].class,
                           Boolean.TYPE,
                           String.class,
                           X509Certificate.class,
                           contentSignerClass,
                           String[].class,
                           ZipFile.class);

                getMetaNameMethod = findMethod(blockClass, GETMETANAME_METHOD);
                writeMethod = findMethod(blockClass, WRITE_METHOD, OutputStream.class);
                
                block = constructor.newInstance(
                        sfg.getJDKSignatureFile(), /* explicit argument on the constructor */
                        privateKey,
                        certChain,
                        externalSF,
                        null,
                        null,
                        null,
                        null,
                        zipFile);
            }

            public String getMetaName() throws IllegalAccessException, InvocationTargetException {
                return (String) getMetaNameMethod.invoke(block);
            }

            public void write(OutputStream os) throws IllegalAccessException, InvocationTargetException {
                writeMethod.invoke(block, os);
            }
        }
    }
}
