/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.appclient.jws.boot;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Arrays;
import java.util.jar.JarFile;

/**
 * Class Path manager for Java Web Start-aware ACC running under Java runtime 1.6.
 *
 * @author tjquinn
 */
public class ClassPathManager16 extends ClassPathManager {
    
    /** Class object for the JNLPClassLoader class */
    private Class jnlpClassLoaderClass;

    /** Method object for the getJarFile method on JNLPClassLoader - only in 1.6 and later */
    private Method getJarFileMethod;

    /**
     *Returns a new instance of the class path manager for use under Java 1.6
     *@param loader the Java Web Start-provided class loader
     */
    protected ClassPathManager16(ClassLoader loader) {
        super(loader);
        try {
            prepareIntrospectionInfo();
        } catch (Throwable thr) {
            throw new RuntimeException(thr);
        }
    }

    /**
     *Prepares the reflection-related private variables for later use in 
     *locating classes in JARs.
     *@throws ClassNotFoundException if the JNLPClassLoader class cannot be found
     *@throws NoSuchMethodException if the getJarFile method cannot be found
     */
    private void prepareIntrospectionInfo() throws ClassNotFoundException, NoSuchMethodException {
        jnlpClassLoaderClass = getJNLPClassLoader().loadClass("com.sun.jnlp.JNLPClassLoader");
        getJarFileMethod = jnlpClassLoaderClass.getDeclaredMethod("getJarFile", URL.class);
        getJarFileMethod.setAccessible(true);
    }

    public ClassLoader getParentClassLoader() {
        return getJNLPClassLoader().getParent();
    }

    public File findContainingJar(URL resourceURL) throws IllegalArgumentException, URISyntaxException, MalformedURLException, IllegalAccessException, InvocationTargetException {
        File result = null;
        if (resourceURL != null) {
            /*
             *The URL will be similar to http://host:port/...path-in-server-namespace!resource-spec
             *Extract the part preceding the ! and ask the Java Web Start loader to
             *find the locally-cached JAR file corresponding to that part of the URL.
             */
            URI resourceURI = resourceURL.toURI();
            String ssp = resourceURI.getSchemeSpecificPart();
            String jarOnlySSP = ssp.substring(0, ssp.indexOf('!'));

            URL jarOnlyURL = new URL(jarOnlySSP).toURI().toURL();

            /*
             *Use introspection to invoke the method.  This avoids complications
             *in building the app server under Java 1.5 in which the JNLPClassLoader
             *does not provide the getJarFile method.
             */
            JarFile jarFile = (JarFile) getJarFileMethod.invoke(getJNLPClassLoader(), jarOnlyURL);
            if (jarFile == null) {
                throw new IllegalArgumentException(resourceURL.toExternalForm());
            }
            result = new File(jarFile.getName());
        }
        return result;
    }
}
