/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * Indentation Information:
 * 0. Please (try to) preserve these settings.
 * 1. Tabs are preferred over spaces.
 * 2. In vi/vim -
 *             :set tabstop=4 :set shiftwidth=4 :set softtabstop=4
 * 3. In S1 Studio -
 *             1. Tools->Options->Editor Settings->Java Editor->Tab Size = 4
 *             2. Tools->Options->Indentation Engines->Java Indentation Engine->Expand Tabs to Spaces = False.
 *             3. Tools->Options->Indentation Engines->Java Indentation Engine->Number of Spaces per Tab = 4.
 */

/* 
 * @author byron.nevins@sun.com
 */

package com.sun.enterprise.deployment.backend;
  
import java.io.*;
import java.util.*;
import java.util.logging.*;
import org.apache.jasper.JspC;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.deployment.WebBundleDescriptor;
import com.sun.enterprise.deployment.WebComponentDescriptor;
import com.sun.enterprise.deployment.web.InitializationParameter;
import com.sun.enterprise.deployment.runtime.web.SunWebApp;
import com.sun.enterprise.deployment.runtime.web.WebProperty;
import com.sun.enterprise.deployment.runtime.web.JspConfig;
import com.sun.appserv.server.util.ASClassLoaderUtil; 

public final class JSPCompiler 
{
	public static void compile(File inWebDir, File outWebDir,
                                   WebBundleDescriptor wbd)
            throws IASDeploymentException 
	{
		compile(inWebDir, outWebDir, wbd, null);
	}

	////////////////////////////////////////////////////////////////////////////
	
	public static void compile(File inWebDir, File outWebDir,
                                   WebBundleDescriptor wbd, List classpathList)
            throws IASDeploymentException 
	{
		JspC jspc = new JspC();

		if (classpathList != null) 
		{
			String classpath = getClasspath(classpathList);
		
			if (classpath != null) 
				jspc.setClassPath(classpath);
		}

                // START SJSAS 6311155
                String appName = wbd.getApplication().getName();
                String sysClassPath = ASClassLoaderUtil.getWebModuleClassPath(appName);
                jspc.setSystemClassPath(sysClassPath);
                // END SJSAS 6311155

		verify(inWebDir, outWebDir);

		configureJspc(jspc, wbd);
		jspc.setOutputDir(outWebDir.getAbsolutePath());
		jspc.setUriroot(inWebDir.getAbsolutePath());
		jspc.setCompile(true);
		logger.info(startMessage);

		try 
		{
			jspc.execute();
		} 
		catch (Exception je) 
		{
			throw new IASDeploymentException("JSP Compilation Error: " + je, je);
		}
		finally
		{
			// bnevins 9-9-03 -- There may be no jsp files in this web-module
			// in such a case the code above will create a useless, and possibly
			// problematic empty directory.	 If the directory is empty -- delete
			// the directory.
			
			String[] files = outWebDir.list();
			
			if(files == null || files.length <= 0)
				outWebDir.delete();
			
			logger.info(finishMessage);
		}
	}

	////////////////////////////////////////////////////////////////////////////
	
	private static void verify(File inWebDir, File outWebDir) throws IASDeploymentException 
	{
		// inWebDir must exist, outWebDir must either exist or be creatable
		if (!FileUtils.safeIsDirectory(inWebDir)) 
		{
			throw new IASDeploymentException("inWebDir is not a directory: " + inWebDir);
		}
	 
		if (!FileUtils.safeIsDirectory(outWebDir)) 
		{
			outWebDir.mkdirs();
		
			if (!FileUtils.safeIsDirectory(outWebDir)) 
			{
				throw new IASDeploymentException("outWebDir is not a directory, and it can't be created: " + outWebDir);
			}
		}
	}

	////////////////////////////////////////////////////////////////////////////
	
	private static String getClasspath(List paths) 
	{
		if(paths == null)
			return null;
		
		String classpath = null;

		StringBuffer sb = new StringBuffer();
		boolean first = true;

		for (Iterator it = paths.iterator(); it.hasNext(); ) 
		{
			String path = (String)it.next();

			if (first) 
				first = false;
			else 
				sb.append(File.pathSeparatorChar);

			sb.append(path);
		}

		if (sb.length() > 0) 
			classpath = sb.toString();

		return classpath;	
	}	

	////////////////////////////////////////////////////////////////////////////

        /*
         * Configures the given JspC instance with the jsp-config properties
	 * specified in the sun-web.xml of the web module represented by the
	 * given WebBundleDescriptor.
	 *
	 * @param jspc JspC instance to be configured
	 * @param wbd WebBundleDescriptor of the web module whose sun-web.xml
         * is used to configure the given JspC instance
	 */
        private static void configureJspc(JspC jspc, WebBundleDescriptor wbd) {

	    SunWebApp sunWebApp = wbd.getSunDescriptor();
	    if (sunWebApp == null) {
		return;
	    }

            // START SJSAS 6384538
            if (sunWebApp.sizeWebProperty() > 0) {
                WebProperty[] props = sunWebApp.getWebProperty();
                for (int i = 0; i < props.length; i++) {
                    String pName = props[i].getAttributeValue("name");
                    String pValue = props[i].getAttributeValue("value");
                    if (pName == null || pValue == null) {
                        throw new IllegalArgumentException(
                            "Missing sun-web-app property name or value");
                    }
                    if ("enableTldValidation".equals(pName)) {
                        jspc.setIsValidationEnabled(
                            Boolean.valueOf(pValue).booleanValue());
                    }
                }
            }
            // END SJSAS 6384538

            // START SJSAS 6170435
            /*
             * Configure JspC with the init params of the JspServlet
             */
            Set set = wbd.getWebComponentDescriptorsSet();         
            if (!set.isEmpty()) {
                Iterator<WebComponentDescriptor> iterator = set.iterator();
                while (iterator.hasNext()) {
                    WebComponentDescriptor webComponentDesc = iterator.next();
                    if ("jsp".equals(webComponentDesc.getCanonicalName())) {
                        Enumeration<InitializationParameter> en
                            = webComponentDesc.getInitializationParameters();
                        if (en != null) {
                            while (en.hasMoreElements()) {
                                InitializationParameter initP = en.nextElement();
                                configureJspc(jspc,
                                              initP.getName(),
                                              initP.getValue());
                            }
                        }
                        break;
                    }
                }
            }
            // END SJSAS 6170435

            /*
             * Configure JspC with jsp-config properties from sun-web.xml,
             * which override JspServlet init params of the same name.
             */
            JspConfig jspConfig = sunWebApp.getJspConfig();
            if (jspConfig == null) {
                return;
            }
            WebProperty[] props = jspConfig.getWebProperty();
            for (int i=0; props!=null && i<props.length; i++) {
                configureJspc(jspc,
                              props[i].getAttributeValue("name"),
                              props[i].getAttributeValue("value"));
            }
        }


        /*
         * Configures the given JspC instance with the given property name
         * and value.
         *
         * @jspc The JspC instance to configure
         * @pName The property name
         * @pValue The property value
         */
        private static void configureJspc(JspC jspc, String pName,
                                          String pValue) {

            if (pName == null || pValue == null) {
                throw new IllegalArgumentException(
                    "Null property name or value");
            }

            if ("xpoweredBy".equals(pName)) {
                jspc.setXpoweredBy(Boolean.valueOf(pValue).booleanValue());
            } else if ("classdebuginfo".equals(pName)) {
                jspc.setClassDebugInfo(Boolean.valueOf(pValue).booleanValue());
            } else if ("enablePooling".equals(pName)) {
                jspc.setPoolingEnabled(Boolean.valueOf(pValue).booleanValue());
            } else if ("ieClassId".equals(pName)) {
                jspc.setIeClassId(pValue);
            } else if ("trimSpaces".equals(pName)) {
                jspc.setTrimSpaces(Boolean.valueOf(pValue).booleanValue());
            } else if ("genStrAsCharArray".equals(pName)) {
                jspc.setGenStringAsCharArray(
                    Boolean.valueOf(pValue).booleanValue());
            } else if ("errorOnUseBeanInvalidClassAttribute".equals(pName)) {
                jspc.setErrorOnUseBeanInvalidClassAttribute(
                    Boolean.valueOf(pValue).booleanValue());
            } else if ("ignoreJspFragmentErrors".equals(pName)) {
                jspc.setIgnoreJspFragmentErrors(
                    Boolean.valueOf(pValue).booleanValue());
            }
        }


	////////////////////////////////////////////////////////////////////////////
	
	private static final String startMessage =
        getStringManager().getStringWithDefault("enterprise.deployment.backend.start_jspc", "Beginning JSP Precompile...");
	private static final String finishMessage =
        getStringManager().getStringWithDefault("enterprise.deployment.backend.finish_jspc", "Finished JSP Precompile");
	private static final Logger logger = DeploymentLogger.get();

	////////////////////////////////////////////////////////////////////////////
	
        private static StringManager
    getStringManager()  {
        return StringManager.getManager( JSPCompiler.class );
    }
}
